import os
import sys
from dotenv import load_dotenv
from flask import Flask
from flask_mail import Mail, Message
import time

# Cargar variables de entorno desde .env
load_dotenv()

# Crear una aplicación Flask mínima para probar el correo
app = Flask(__name__)

# Configurar la aplicación con los valores del .env
app.config['MAIL_SERVER'] = os.environ.get('MAIL_SERVER')
app.config['MAIL_PORT'] = int(os.environ.get('MAIL_PORT') or 25)

# Corregir la configuración de TLS/SSL - no pueden estar ambos activos
# Si el puerto es 465, generalmente se usa SSL. Si es 587, se usa TLS.
use_ssl = os.environ.get('MAIL_USE_SSL') is not None
use_tls = os.environ.get('MAIL_USE_TLS') is not None

# Si ambos están configurados, dar prioridad a SSL si el puerto es 465
if use_ssl and use_tls:
    if app.config['MAIL_PORT'] == 465:
        use_tls = False
    else:
        use_ssl = False

app.config['MAIL_USE_TLS'] = use_tls
app.config['MAIL_USE_SSL'] = use_ssl
app.config['MAIL_USERNAME'] = os.environ.get('MAIL_USERNAME')
app.config['MAIL_PASSWORD'] = os.environ.get('MAIL_PASSWORD')
app.config['MAIL_DEFAULT_SENDER'] = os.environ.get('MAIL_DEFAULT_SENDER') or os.environ.get('MAIL_USERNAME')
app.config['MAIL_TIMEOUT'] = int(os.environ.get('MAIL_TIMEOUT') or 30)  # Aumentamos el timeout por defecto

# Inicializar la extensión de correo
mail = Mail(app)

def print_config():
    """Imprime la configuración actual de correo (sin mostrar la contraseña)"""
    print("\n=== Configuración actual de correo ===")
    print(f"MAIL_SERVER: {app.config['MAIL_SERVER']}")
    print(f"MAIL_PORT: {app.config['MAIL_PORT']}")
    print(f"MAIL_USE_TLS: {app.config['MAIL_USE_TLS']}")
    print(f"MAIL_USE_SSL: {app.config['MAIL_USE_SSL']}")
    print(f"MAIL_USERNAME: {app.config['MAIL_USERNAME']}")
    print(f"MAIL_PASSWORD: {'*' * 8 if app.config['MAIL_PASSWORD'] else 'No configurado'}")
    print(f"MAIL_DEFAULT_SENDER: {app.config['MAIL_DEFAULT_SENDER']}")
    print(f"MAIL_TIMEOUT: {app.config['MAIL_TIMEOUT']} segundos")
    print("=====================================\n")

def send_test_email(recipient=None):
    """
    Envía un correo electrónico de prueba
    
    Args:
        recipient: Dirección de correo del destinatario
    """
    if not recipient:
        recipient = input("Ingresa la dirección de correo del destinatario: ")
    
    subject = "Prueba de correo - OLE Consulting"
    
    # Usar el contexto de aplicación para crear y enviar el mensaje
    with app.app_context():
        # Crear el mensaje
        msg = Message(
            subject=subject,
            recipients=[recipient],
        body=f"""
Este es un correo de prueba enviado desde la aplicación OLE Consulting.
Fecha y hora: {time.strftime('%Y-%m-%d %H:%M:%S')}

Si has recibido este correo, la configuración SMTP está funcionando correctamente.

Detalles técnicos:
- Servidor: {app.config['MAIL_SERVER']}
- Puerto: {app.config['MAIL_PORT']}
- TLS: {app.config['MAIL_USE_TLS']}
- SSL: {app.config['MAIL_USE_SSL']}
- Timeout: {app.config['MAIL_TIMEOUT']} segundos

Saludos,
Equipo de OLE Consulting
        """,
        html=f"""
<h2>Prueba de correo - OLE Consulting</h2>
<p>Este es un correo de prueba enviado desde la aplicación OLE Consulting.</p>
<p>Fecha y hora: <strong>{time.strftime('%Y-%m-%d %H:%M:%S')}</strong></p>
<p>Si has recibido este correo, la configuración SMTP está funcionando correctamente.</p>
<h3>Detalles técnicos:</h3>
<ul>
    <li>Servidor: {app.config['MAIL_SERVER']}</li>
    <li>Puerto: {app.config['MAIL_PORT']}</li>
    <li>TLS: {app.config['MAIL_USE_TLS']}</li>
    <li>SSL: {app.config['MAIL_USE_SSL']}</li>
    <li>Timeout: {app.config['MAIL_TIMEOUT']} segundos</li>
</ul>
<p>Saludos,<br>
Equipo de OLE Consulting</p>
        """
    )
    
    print(f"\nEnviando correo de prueba a {recipient}...")
    start_time = time.time()
    
    try:
        with app.app_context():
            mail.send(msg)
        elapsed_time = time.time() - start_time
        print(f"✅ Correo enviado exitosamente en {elapsed_time:.2f} segundos")
        return True
    except Exception as e:
        elapsed_time = time.time() - start_time
        print(f"❌ Error al enviar el correo después de {elapsed_time:.2f} segundos:")
        print(f"   {type(e).__name__}: {str(e)}")
        return False

def test_connection():
    """Prueba la conexión al servidor SMTP sin enviar un correo"""
    import smtplib
    
    server = app.config['MAIL_SERVER']
    port = app.config['MAIL_PORT']
    use_ssl = app.config['MAIL_USE_SSL']
    use_tls = app.config['MAIL_USE_TLS']
    username = app.config['MAIL_USERNAME']
    password = app.config['MAIL_PASSWORD']
    timeout = app.config['MAIL_TIMEOUT']
    
    print(f"\nProbando conexión a {server}:{port}...")
    start_time = time.time()
    
    try:
        if use_ssl:
            smtp = smtplib.SMTP_SSL(server, port, timeout=timeout)
        else:
            smtp = smtplib.SMTP(server, port, timeout=timeout)
        
        smtp.ehlo()
        
        if use_tls and not use_ssl:
            smtp.starttls()
            smtp.ehlo()
        
        if username and password:
            smtp.login(username, password)
        
        smtp.quit()
        
        elapsed_time = time.time() - start_time
        print(f"✅ Conexión exitosa en {elapsed_time:.2f} segundos")
        return True
    except Exception as e:
        elapsed_time = time.time() - start_time
        print(f"❌ Error de conexión después de {elapsed_time:.2f} segundos:")
        print(f"   {type(e).__name__}: {str(e)}")
        return False

def main():
    """Función principal del script"""
    print("\n=== Prueba de Configuración de Correo - OLE Consulting ===\n")
    
    # Verificar si hay variables de entorno necesarias
    if not app.config['MAIL_SERVER']:
        print("❌ Error: No se ha configurado MAIL_SERVER en el archivo .env")
        return
    
    # Mostrar la configuración actual
    print_config()
    
    # Menú de opciones
    while True:
        print("\nOpciones:")
        print("1. Probar conexión al servidor SMTP")
        print("2. Enviar correo de prueba")
        print("3. Mostrar configuración actual")
        print("4. Salir")
        
        choice = input("\nSelecciona una opción (1-4): ")
        
        if choice == '1':
            test_connection()
        elif choice == '2':
            send_test_email()
        elif choice == '3':
            print_config()
        elif choice == '4':
            print("\n¡Hasta luego!")
            break
        else:
            print("\n❌ Opción no válida. Por favor, selecciona una opción del 1 al 4.")

if __name__ == "__main__":
    # Inicializar la aplicación Flask con el contexto adecuado
    with app.app_context():
        # Si se pasa un argumento, se asume que es la dirección de correo para enviar directamente
        if len(sys.argv) > 1:
            recipient = sys.argv[1]
            print(f"Enviando correo de prueba a {recipient} (modo automático)")
            print_config()
            send_test_email(recipient)
        else:
            # Modo interactivo
            main()
