"""
Script de prueba para la generación de imágenes con DALL·E
"""
import os
import sys
import logging
import requests
from datetime import datetime
from werkzeug.utils import secure_filename
from dotenv import load_dotenv

# Configurar logging
logging.basicConfig(level=logging.INFO, 
                   format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

# Cargar variables de entorno
load_dotenv()

# Usar la clave API proporcionada directamente, ignorando variables de entorno
OPENAI_API_KEY = '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'

def generate_test_image(title="Energía renovable en España"):
    """
    Prueba la generación de imágenes con DALL·E y el guardado correcto
    """
    if not OPENAI_API_KEY:
        logger.error("OPENAI_API_KEY no está configurada. Por favor, configúrala en las variables de entorno.")
        return None
    
    logger.info(f"Generando imagen de prueba para: {title}")
    
    # Crear un prompt detallado para la imagen
    prompt = f"""Crea una imagen fotorrealista y profesional para un artículo de blog sobre 
'{title}'. Debe ser una fotografía conceptual relacionada con energía y sostenibilidad, 
con estilo limpio, tecnológico y profesional. Formato horizontal 16:9. Sin texto ni letras."""
    
    # Endpoint de la API de OpenAI para DALL-E
    url = "https://api.openai.com/v1/images/generations"
    headers = {
        "Content-Type": "application/json",
        "Authorization": f"Bearer {OPENAI_API_KEY}"
    }
    
    # Payload para la solicitud DALL-E
    payload = {
        "model": "dall-e-3",
        "prompt": prompt,
        "n": 1,
        "size": "1792x1024",
        "quality": "standard",
        "style": "natural"
    }
    
    try:
        logger.info(f"Enviando solicitud a DALL·E...")
        response = requests.post(url, headers=headers, json=payload, timeout=30)
        response.raise_for_status()
        
        data = response.json()
        if "data" in data and len(data["data"]) > 0 and "url" in data["data"][0]:
            img_url = data["data"][0]["url"]
            logger.info(f"URL de imagen generada: {img_url}")
            
            # Descargar la imagen generada
            img_response = requests.get(img_url, timeout=15)
            img_response.raise_for_status()
            img_bytes = img_response.content
            logger.info(f"Imagen descargada: {len(img_bytes)} bytes")
            
            # Asegurarse de que el directorio existe
            upload_dir = os.path.abspath('app/static/uploads/ai')
            os.makedirs(upload_dir, exist_ok=True)
            logger.info(f"Directorio para guardar: {upload_dir}")
            
            # Crear un nombre de archivo seguro con timestamp
            timestamp = datetime.utcnow().strftime('%Y%m%d%H%M%S')
            safe_title = secure_filename(title[:30] if title else "energia")
            filename = f"dalle_{timestamp}_{safe_title}.png"
            file_path = os.path.join(upload_dir, filename)
            
            # Guardar la imagen
            with open(file_path, 'wb') as img_file:
                img_file.write(img_bytes)
            logger.info(f"Imagen guardada en: {file_path}")
            
            # Devolver la ruta correcta para la web
            web_path = f"/static/uploads/ai/{filename}"
            logger.info(f"Ruta de acceso web: {web_path}")
            
            # Para fines de prueba, verificar si el archivo existe
            if os.path.exists(file_path):
                logger.info(f"✅ ÉXITO: El archivo existe físicamente en disco")
                logger.info(f"Tamaño del archivo: {os.path.getsize(file_path)} bytes")
                logger.info(f"Permisos del archivo: {oct(os.stat(file_path).st_mode)[-3:]}")
            else:
                logger.error(f"❌ ERROR: El archivo no existe en disco después de guardarlo")
            
            return web_path
        else:
            logger.error("Formato de respuesta de DALL-E inesperado")
            return None
    except requests.exceptions.HTTPError as http_err:
        logger.error(f"Error HTTP al generar imagen: {http_err}")
        if response.status_code == 429:
            logger.error("Se ha alcanzado el límite de tasa de la API de OpenAI")
        elif response.status_code == 401:
            logger.error("Error de autenticación con la API de OpenAI")
        try:
            error_data = response.json()
            logger.error(f"Detalles del error: {error_data}")
        except:
            pass
    except Exception as e:
        logger.error(f"Error al generar imagen: {str(e)}")
    
    return None

def fix_database_image_paths():
    """
    Corregir las rutas de imágenes en la base de datos
    """
    try:
        # Importar la aplicación Flask y sus modelos
        from app import create_app
        from app.extensions import db
        from app.models.article import Article
        
        # Crear una instancia de la aplicación y establecer el contexto
        app = create_app()
        with app.app_context():
            logger.info("Contexto de aplicación establecido correctamente")
            
            # Buscar artículos con rutas de imágenes incorrectas
            articles_to_fix = Article.query.filter(Article.featured_image.like('/blog/uploads/ai/%')).all()
            count = len(articles_to_fix)
            
            # Buscar también rutas que no comiencen con /static/
            other_articles = Article.query.filter(
                Article.featured_image.isnot(None),
                ~Article.featured_image.like('/static/%')
            ).all()
            
            # Acumular todos los artículos que necesitan correcciones
            all_articles_to_fix = list(articles_to_fix)
            all_articles_to_fix.extend(other_articles)
            total_count = len(all_articles_to_fix)
            
            if total_count == 0:
                logger.info("No se encontraron artículos con rutas incorrectas en la base de datos.")
                return
            
            logger.info(f"Se encontraron {total_count} artículos con rutas incorrectas para corregir.")
            
            # Corregir las rutas
            for article in all_articles_to_fix:
                old_path = article.featured_image
                new_path = old_path
                
                # Caso 1: Rutas que comienzan con /blog/uploads/ai/
                if old_path.startswith('/blog/uploads/ai/'):
                    new_path = old_path.replace('/blog/uploads/ai/', '/static/uploads/ai/')
                
                # Caso 2: Rutas con backslash sin prefijo /static/
                elif '\\' in old_path or old_path.startswith('uploads\\'):
                    # Extraer solo el nombre del archivo
                    filename = os.path.basename(old_path.replace('\\', '/'))
                    new_path = f"/static/uploads/ai/{filename}"
                
                # Caso 3: Rutas con slash sin prefijo /static/
                elif old_path.startswith('uploads/'):
                    # Extraer solo el nombre del archivo
                    filename = os.path.basename(old_path)
                    new_path = f"/static/uploads/ai/{filename}"
                
                # Si la ruta ha cambiado, actualizar el artículo
                if new_path != old_path:
                    article.featured_image = new_path
                    logger.info(f"Artículo {article.id}: {old_path} -> {new_path}")
            
            # Guardar los cambios
            db.session.commit()
            logger.info(f"✅ ÉXITO: Se corrigieron {total_count} rutas de imágenes en la base de datos.")
    except Exception as e:
        logger.error(f"Error al corregir rutas de imágenes: {str(e)}")

def fix_blog_template_paths():
    """
    Busca los archivos de plantilla que podrían estar usando rutas incorrectas
    y muestra información sobre ellos
    """
    try:
        logger.info("Buscando plantillas con posibles rutas incorrectas...")
        template_dir = os.path.abspath('app/templates')
        
        # Lista para almacenar archivos con potenciales problemas
        problematic_files = []
        
        # Funciones para buscar patrones problemáticos en archivos
        def search_in_file(file_path):
            try:
                with open(file_path, 'r', encoding='utf-8') as f:
                    content = f.read()
                    if '/blog/uploads/ai/' in content:
                        return True, "Contiene ruta '/blog/uploads/ai/'"
                    return False, ""
            except Exception as e:
                return False, f"Error al leer archivo: {str(e)}"
        
        # Recorrer directorio de plantillas recursivamente
        for root, _, files in os.walk(template_dir):
            for file in files:
                if file.endswith('.html'):
                    file_path = os.path.join(root, file)
                    is_problematic, reason = search_in_file(file_path)
                    if is_problematic:
                        rel_path = os.path.relpath(file_path, os.path.abspath('app'))
                        problematic_files.append((rel_path, reason))
        
        # Mostrar resultados
        if problematic_files:
            logger.info(f"Se encontraron {len(problematic_files)} plantillas con posibles problemas:")
            for file_path, reason in problematic_files:
                logger.info(f"Archivo: {file_path} - Razón: {reason}")
        else:
            logger.info("No se encontraron plantillas con problemas de rutas.")
            
    except Exception as e:
        logger.error(f"Error al buscar plantillas: {str(e)}")


def check_file_access():
    """
    Verifica si las imágenes en la carpeta de uploads son accesibles y tienen permisos adecuados
    """
    upload_dir = os.path.abspath('app/static/uploads/ai')
    
    if not os.path.exists(upload_dir):
        logger.warning(f"El directorio {upload_dir} no existe. Creándolo...")
        try:
            os.makedirs(upload_dir, exist_ok=True)
            logger.info(f"Directorio creado exitosamente: {upload_dir}")
        except Exception as e:
            logger.error(f"Error al crear directorio: {str(e)}")
            return
    
    try:
        logger.info(f"Verificando archivos en {upload_dir}...")
        files = os.listdir(upload_dir)
        logger.info(f"Total de archivos: {len(files)}")
        
        for i, filename in enumerate(files[:5]):  # Mostrar solo los primeros 5 para no saturar
            file_path = os.path.join(upload_dir, filename)
            if os.path.isfile(file_path):
                stats = os.stat(file_path)
                permissions = oct(stats.st_mode)[-3:]
                logger.info(f"Archivo {i+1}: {filename} - Tamaño: {stats.st_size} bytes - Permisos: {permissions}")
    except Exception as e:
        logger.error(f"Error al verificar archivos: {str(e)}")


if __name__ == "__main__":
    if len(sys.argv) > 1:
        if sys.argv[1] == "fix-db":
            logger.info("Iniciando corrección de rutas de imágenes en la base de datos...")
            fix_database_image_paths()
        elif sys.argv[1] == "check-templates":
            logger.info("Verificando plantillas con posibles problemas de rutas...")
            fix_blog_template_paths()
        elif sys.argv[1] == "check-files":
            logger.info("Verificando archivos en la carpeta de uploads...")
            check_file_access()
    else:
        logger.info("Iniciando prueba de generación de imagen...")
        # Verificar que la API key de OpenAI esté configurada
        if OPENAI_API_KEY:
            logger.info("API key de OpenAI disponible")
        else:
            logger.warning("API key de OpenAI no configurada")
            
        image_path = generate_test_image()
        if image_path:
            logger.info(f"Prueba completada con éxito. Ruta de imagen: {image_path}")
        else:
            logger.error("La prueba falló.")
