#!/usr/bin/env python
"""
Script para configurar Poppler desde la carpeta vendor del proyecto.
Este script detecta si Poppler está disponible en la carpeta vendor
y configura la variable de entorno POPPLER_PATH en consecuencia.
"""
import os
import sys
import logging
from pathlib import Path
from dotenv import load_dotenv, set_key

# Configurar logging
logging.basicConfig(level=logging.INFO, format='%(levelname)s: %(message)s')

def setup_poppler():
    """Configura Poppler desde la carpeta vendor del proyecto."""
    # Obtener la ruta base del proyecto
    base_dir = os.path.dirname(os.path.abspath(__file__))
    
    # Definir la ruta a Poppler en la carpeta vendor
    vendor_poppler_path = os.path.join(base_dir, 'vendor', 'poppler')
    
    # Verificar si existe la carpeta vendor/poppler
    if os.path.exists(vendor_poppler_path) and os.path.isdir(vendor_poppler_path):
        # Verificar si contiene los ejecutables necesarios
        poppler_exes = ['pdftoppm.exe', 'pdfinfo.exe']
        has_exes = all(os.path.exists(os.path.join(vendor_poppler_path, exe)) for exe in poppler_exes)
        
        if has_exes:
            logging.info(f"Poppler encontrado en: {vendor_poppler_path}")
            
            # Actualizar la variable de entorno POPPLER_PATH
            os.environ['POPPLER_PATH'] = vendor_poppler_path
            
            # Actualizar el archivo .env
            env_file = os.path.join(base_dir, '.env')
            if os.path.exists(env_file):
                # Cargar variables de entorno existentes
                load_dotenv(env_file)
                
                # Actualizar o añadir la variable POPPLER_PATH
                set_key(env_file, 'POPPLER_PATH', vendor_poppler_path)
                logging.info(f"Archivo .env actualizado con POPPLER_PATH={vendor_poppler_path}")
            else:
                # Crear un nuevo archivo .env si no existe
                with open(env_file, 'a') as f:
                    f.write(f"\nPOPPLER_PATH={vendor_poppler_path}\n")
                logging.info(f"Archivo .env creado con POPPLER_PATH={vendor_poppler_path}")
            
            return True
    
    # Si no se encuentra Poppler en vendor, buscar en la ruta del sistema
    logging.warning("No se encontró Poppler en la carpeta vendor del proyecto.")
    
    # Intentar usar la variable de entorno existente
    if 'POPPLER_PATH' in os.environ and os.path.exists(os.environ['POPPLER_PATH']):
        logging.info(f"Usando Poppler desde la variable de entorno: {os.environ['POPPLER_PATH']}")
        return True
    
    logging.warning("No se pudo encontrar Poppler. Por favor, copia los archivos de Poppler a la carpeta vendor/poppler.")
    return False

if __name__ == "__main__":
    if setup_poppler():
        print("\n✅ Poppler configurado correctamente")
        sys.exit(0)
    else:
        print("\n❌ No se pudo configurar Poppler")
        sys.exit(1)
