"""seed_rates_from_json

Revision ID: a45ff3b38f5b
Revises: d1ebcedb7000
Create Date: 2025-07-10 05:17:51.577945

"""
from alembic import op
import sqlalchemy as sa
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import Session
import json
import os
from datetime import datetime

# Importamos el modelo Rate directamente para evitar problemas de importación circular
Base = declarative_base()

class Rate(Base):
    __tablename__ = 'rates'
    
    id = sa.Column(sa.Integer, primary_key=True)
    name = sa.Column(sa.String(120), nullable=False)
    tariff_type = sa.Column(sa.String(50), nullable=False)
    energy_prices = sa.Column(sa.JSON, nullable=False)
    power_prices = sa.Column(sa.JSON, nullable=False)
    valid_from = sa.Column(sa.Date, nullable=False)
    valid_until = sa.Column(sa.Date, nullable=True)
    notes = sa.Column(sa.Text, nullable=True)
    supplier = sa.Column(sa.String(120), nullable=True)
    created_at = sa.Column(sa.DateTime, default=datetime.utcnow)
    updated_at = sa.Column(sa.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)


# revision identifiers, used by Alembic.
revision = 'a45ff3b38f5b'
down_revision = 'd1ebcedb7000'
branch_labels = None
depends_on = None


def upgrade():
    # Crear una conexión a la base de datos
    bind = op.get_bind()
    session = Session(bind=bind)
    
    try:
        # Verificar si ya hay tarifas en la base de datos
        if session.query(Rate).count() > 0:
            print("Ya existen tarifas en la base de datos. Omitiendo migración.")
            return
        
        # Obtener la ruta del archivo JSON
        # Asumimos que el archivo está en app/data/best_rates.json
        # Calculamos la ruta relativa desde el archivo de migración
        current_dir = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
        rates_path = os.path.join(current_dir, 'app', 'data', 'best_rates.json')
        
        if not os.path.exists(rates_path):
            print(f"No se encontró el archivo de tarifas en {rates_path}")
            return
        
        # Cargar tarifas desde el archivo JSON
        with open(rates_path, 'r', encoding='utf-8') as file:
            data = json.load(file)
            rates = data.get('rates', [])
        
        if not rates:
            print("No se encontraron tarifas en el archivo JSON.")
            return
            
        print(f"Se encontraron {len(rates)} tarifas en el archivo JSON.")
        
        # Migrar cada tarifa a la base de datos
        for rate_data in rates:
            # Convertir fechas de string a objeto Date
            valid_from = datetime.strptime(rate_data.get('valid_from'), '%Y-%m-%d').date() if rate_data.get('valid_from') else None
            valid_until = datetime.strptime(rate_data.get('valid_until'), '%Y-%m-%d').date() if rate_data.get('valid_until') else None
            
            # Crear objeto Rate
            rate = Rate(
                name=rate_data.get('name'),
                supplier=rate_data.get('supplier'),
                tariff_type=rate_data.get('tariff_type'),
                energy_prices=rate_data.get('energy_prices', {}),
                power_prices=rate_data.get('power_prices', {}),
                valid_from=valid_from,
                valid_until=valid_until,
                notes=rate_data.get('notes', '')
            )
            
            # Añadir a la sesión
            session.add(rate)
        
        # Commit de todas las tarifas
        session.commit()
        print(f"Se migraron {len(rates)} tarifas a la base de datos correctamente.")
        
    except Exception as e:
        session.rollback()
        print(f"Error al migrar tarifas: {e}")
        raise
    finally:
        session.close()


def downgrade():
    # Eliminar todas las tarifas importadas desde el archivo JSON
    bind = op.get_bind()
    session = Session(bind=bind)
    
    try:
        # Eliminar todas las tarifas
        # Nota: En un entorno de producción, podrías querer ser más selectivo
        # sobre qué tarifas eliminar, por ejemplo, basándote en fechas o IDs específicos
        deleted_count = session.query(Rate).delete()
        session.commit()
        print(f"Se eliminaron {deleted_count} tarifas de la base de datos.")
    except Exception as e:
        session.rollback()
        print(f"Error al eliminar tarifas: {e}")
        raise
    finally:
        session.close()
