#!/usr/bin/env python3
"""
Script de migración para cambiar del sistema DeepSeek único al sistema multi-proveedor.
Actualiza automáticamente las importaciones sin interrumpir el servicio.
"""
import os
import re
import shutil
from pathlib import Path

def backup_file(file_path):
    """Crea una copia de seguridad del archivo."""
    backup_path = f"{file_path}.backup"
    shutil.copy2(file_path, backup_path)
    print(f"✅ Backup creado: {backup_path}")
    return backup_path

def update_imports_in_file(file_path):
    """Actualiza las importaciones en un archivo específico."""
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
        
        original_content = content
        
        # Reemplazar importaciones de llm_client por llm_multi_provider
        patterns = [
            (r'from app\.services\.llm_client import extract_with_llm', 
             'from app.services.llm_multi_provider import extract_with_llm'),
            (r'from app\.services import llm_client', 
             'from app.services import llm_multi_provider'),
            (r'import app\.services\.llm_client', 
             'import app.services.llm_multi_provider'),
        ]
        
        for old_pattern, new_import in patterns:
            content = re.sub(old_pattern, new_import, content)
        
        if content != original_content:
            backup_backup_path = backup_file(file_path)
            
            with open(file_path, 'w', encoding='utf-8') as f:
                f.write(content)
            
            print(f"✅ Actualizado: {file_path}")
            return True
        else:
            print(f"⏭️  Sin cambios: {file_path}")
            return False
            
    except Exception as e:
        print(f"❌ Error procesando {file_path}: {e}")
        return False

def find_files_using_llm_client():
    """Encuentra todos los archivos que usan llm_client."""
    project_root = Path(__file__).parent
    files_to_update = []
    
    # Buscar en directorios relevantes
    search_dirs = [
        project_root / "app" / "services",
        project_root / "app" / "blueprints",
        project_root / "app"
    ]
    
    for search_dir in search_dirs:
        if search_dir.exists():
            for py_file in search_dir.rglob("*.py"):
                try:
                    with open(py_file, 'r', encoding='utf-8') as f:
                        content = f.read()
                    
                    if 'llm_client' in content and 'extract_with_llm' in content:
                        files_to_update.append(py_file)
                        
                except Exception as e:
                    print(f"Error leyendo {py_file}: {e}")
    
    return files_to_update

def main():
    print("=== MIGRACIÓN A SISTEMA MULTI-PROVEEDOR ===\n")
    
    # 1. Encontrar archivos que usan llm_client
    print("1. Buscando archivos que usan llm_client...")
    files_to_update = find_files_using_llm_client()
    
    if not files_to_update:
        print("ℹ️  No se encontraron archivos que requieran actualización")
        return
    
    print(f"📁 Encontrados {len(files_to_update)} archivos para actualizar:")
    for file_path in files_to_update:
        print(f"   - {file_path}")
    
    # 2. Actualizar cada archivo
    print("\n2. Actualizando archivos...")
    updated_count = 0
    
    for file_path in files_to_update:
        if update_imports_in_file(file_path):
            updated_count += 1
    
    # 3. Resumen
    print(f"\n=== MIGRACIÓN COMPLETADA ===")
    print(f"✅ {updated_count} archivos actualizados")
    print(f"📁 {len(files_to_update) - updated_count} archivos sin cambios")
    
    print("\n📋 PRÓXIMOS PASOS:")
    print("1. Configura al menos una API KEY (OPENAI_API_KEY o GROQ_API_KEY) en tu .env")
    print("2. Reinicia el servidor Flask")
    print("3. Ejecuta: python test_multi_provider.py")
    print("\n🔄 Para revertir cambios: restaura desde los archivos .backup creados")

if __name__ == "__main__":
    main()
