import os
import sys
import click
from dotenv import load_dotenv
from flask_migrate import Migrate

print(f"DEBUG: Python Executable: {sys.executable}")
print(f"DEBUG: Python Path: {sys.path}")

# Cargar variables de entorno desde .env ANTES de importar la app
# Esto es crucial si config.py o __init__.py dependen de ellas al cargar.
dotenv_path = os.path.join(os.path.dirname(__file__), '.env')
if os.path.exists(dotenv_path):
    # override=True asegura que los valores del .env prevalezcan sobre variables del SO vacías o distintas
    load_dotenv(dotenv_path, override=True)

from app import create_app, db
# Importa tus modelos aquí para que Flask-Migrate los detecte
from app.models import User, Article, InvoiceLead # Importamos directamente los modelos

app = create_app(os.getenv('FLASK_ENV') or 'default')
migrate = Migrate(app, db) # Esta instancia de Migrate es la que se usa para los comandos

# Aseguramos que la instancia de la aplicación y la migración estén disponibles

@app.shell_context_processor
def make_shell_context():
    """Define el contexto para el shell de Flask."""
    return dict(db=db, User=User, Article=Article, InvoiceLead=InvoiceLead)

@app.cli.command('init_db')
def init_db_command():
    """Inicializa la base de datos."""
    db.create_all()
    print('Base de datos inicializada.')

@app.cli.command('seed_db')
def seed_db_command():
    """Puebla la base de datos con datos iniciales (opcional)."""
    # Ejemplo: Crear un usuario admin
    # from app.models.user import User
    # if not User.query.filter_by(username='admin').first():
    #     admin_user = User(username='admin', email='admin@example.com')
    #     admin_user.set_password('adminpass')
    #     db.session.add(admin_user)
    #     db.session.commit()
    #     print('Usuario admin creado.')
    print('Base de datos poblada (si la función está implementada).')

@app.cli.command('create_admin')
def create_admin_command():
    """Crea un usuario administrador con credenciales predefinidas."""
    from app.models.user import User
    
    username = 'carlo.golik'
    password = '20222456'
    email = f'{username}@oleconsulting.es'

    existing_user = User.query.filter_by(username=username).first()
    if existing_user:
        print(f'El usuario {username} ya existe.')
        if not existing_user.is_admin:
            existing_user.is_admin = True
            db.session.commit()
            print(f'Se han concedido permisos de administrador a {username}.')
        return
    
    admin_user = User(username=username, email=email, is_admin=True)
    admin_user.set_password(password)
    
    db.session.add(admin_user)
    db.session.commit()
    print(f'Usuario administrador {username} creado exitosamente.')
    print('IMPORTANTE: Cambie esta contraseña después del primer inicio de sesión.')

# Para ejecutar con Gunicorn (ej: gunicorn 'manage:app')
# Si usas una factoría, Gunicorn necesita que le pases la app directamente o una forma de obtenerla.
# La variable 'app' ya está definida arriba con create_app().

# Lista de comandos disponibles
@app.cli.command('list_commands')
def list_commands_command():
    """Lista todos los comandos disponibles."""
    print("Comandos disponibles:")
    print(" - flask init_db: Inicializa la base de datos")
    print(" - flask seed_db: Puebla la base de datos con datos iniciales")
    print(" - flask create_admin: Crea un usuario con permisos de administrador")
    print(" - flask create_roles: Crea o actualiza los roles del sistema")
    print(" - flask list_commands: Muestra esta lista de comandos")
    
@app.cli.command('create_roles')
def create_roles_command():
    """Crea o actualiza los roles del sistema."""
    from app.models.role import Role
    
    # Lista de roles predefinidos
    roles = ['Super Admin', 'Agent', 'Blogger', 'Price Analyst', 'Comercial']
    
    for role_name in roles:
        role = Role.query.filter_by(name=role_name).first()
        if not role:
            role = Role(name=role_name)
            if role_name == 'Comercial':
                role.description = 'Puede ver tarifas y leads, pero no puede modificar o eliminar registros.'
            elif role_name == 'Super Admin':
                role.description = 'Acceso completo a todas las funciones.'
            elif role_name == 'Agent':
                role.description = 'Puede ver leads pero no borrarlos ni descargar facturas.'
            elif role_name == 'Blogger':
                role.description = 'Puede crear y gestionar contenido del blog y FAQ.'
            elif role_name == 'Price Analyst':
                role.description = 'Puede gestionar las tarifas de energía.'
                
            db.session.add(role)
            print(f'Rol "{role_name}" creado.')
        else:
            print(f'El rol "{role_name}" ya existe.')
            
    db.session.commit()
    print('Roles actualizados correctamente.')
    
@app.cli.command('list_users')
def list_users_command():
    """Lista todos los usuarios y sus roles."""
    from app.blueprints.admin.commands import list_users
    list_users()
    
@app.cli.command('assign_role')
@click.argument('user_id', type=int)
@click.argument('role_name', type=str)
def assign_role_command(user_id, role_name):
    """Asigna un rol específico a un usuario por su ID.
    
    Ejemplo: flask assign_role 1 'Super Admin'
    """
    from app.blueprints.admin.commands import assign_role_to_user
    assign_role_to_user(user_id, role_name)


if __name__ == '__main__':
    # Esto es para desarrollo local con `python manage.py`
    # Para producción, usa un servidor WSGI como Gunicorn.
    # Cambiamos puerto a 5050 porque 5000 y 5173 están en uso
    # app.run(debug=True) # Línea original o similar
    app.run(host='0.0.0.0', port=5050, debug=app.config.get('DEBUG', True))
    app.run(host='0.0.0.0', port=5050, debug=app.config.get('DEBUG', True))
