"""
Script para corregir problemas de indentación y patrones en tariff_detector.py
"""
import re
import os
import shutil
import sys

def fix_energy_price_section(file_path):
    """
    Lee el archivo tariff_detector.py, corrige la sección de detección de precios de energía
    y guarda una versión corregida del archivo.
    """
    # Hacer una copia de seguridad del archivo original
    backup_path = file_path + '.bak'
    shutil.copy2(file_path, backup_path)
    print(f"Copia de seguridad creada en: {backup_path}")
    
    # Leer el contenido del archivo
    with open(file_path, 'r', encoding='utf-8', errors='ignore') as f:
        content = f.read()
    
    # Buscar la sección problemática - comienzo después de la sección general de precios de energía
    start_marker = "            # Búsqueda adicional para periodos no encontrados"
    end_marker = "            # Ahora buscamos consumos de energía para cada periodo"
    
    # Obtener las partes del archivo
    try:
        before_section = content.split(start_marker)[0]
        # Intentar encontrar la sección final
        remaining = content.split(start_marker)[1]
        
        try:
            after_section = remaining.split(end_marker)[1]
            end_found = True
        except:
            # Si no hay un marcador de fin claro, intentamos encontrar la siguiente sección principal
            end_patterns = [
                "            # Ahora buscamos consumos", 
                "            # Extracción de precios de potencia",
                "        # Final del procesamiento"
            ]
            end_found = False
            for pattern in end_patterns:
                if pattern in remaining:
                    parts = remaining.split(pattern, 1)
                    after_section = pattern + parts[1]
                    end_found = True
                    break
            
            if not end_found:
                # Si no encontramos ningún marcador de fin, asumimos que el problema
                # está contenido en un bloque específico y preservamos el resto
                potential_end = re.search(r'\n\s{8,12}[#][^#]', remaining)
                if potential_end:
                    pos = potential_end.start()
                    after_section = remaining[pos:]
                else:
                    # No encontramos un punto de fin claro, solo reemplazamos el bloque problemático
                    print("ADVERTENCIA: No se encontró un marcador de fin claro. Se reemplazará solo el bloque problemático.")
                    after_section = ""
        
        # Código corregido para la sección de precios de energía
        fixed_section = """            # Búsqueda adicional para periodos no encontrados
            for period_key in tariff_structure.get('energy', {}).keys():
                if period_key not in self.energy_prices or not self.energy_prices[period_key]:
                    logging.info(f"No se encontró precio de energía para {period_key} en formato estándar, buscando formato específico...")
                    
                    # Buscar en formato específico de factura (Ex NNN kWh x N,NNNN €/kWh)
                    period_num = period_key[1:]  # Extrae el número (1, 2, 3) de P1, P2, P3
                    # Simplificamos el patrón para hacerlo más tolerante
                    specific_pattern = rf'E{period_num}[^\d]+(\d+[,.]\d+)[\s]*kWh[\s]*[xX][\s]*(\d+[,.]\d+)'
                    specific_match = re.search(specific_pattern, self.text, re.IGNORECASE)
                    logging.info(f"Buscando con patrón: {specific_pattern}")
                    
                    # Intento con patrones más simples adaptados a facturas españolas
                    if not specific_match:
                        # Usar patrón más simple sin corchetes incorrectos
                        simple_pattern = rf'E{period_num}\D+(\d+[,.]\d+)\D+kWh\D+[xX]\D+(\d+[,.]\d+)'
                        specific_match = re.search(simple_pattern, self.text, re.IGNORECASE)
                        logging.info(f"Segundo intento con patrón: {simple_pattern}")
                        
                    # Un tercer intento con patrón aún más simple
                    if not specific_match:
                        very_simple = rf'E{period_num}.*?(\d+[,.]\d+).*?kWh.*?[xX].*?(\d+[,.]\d+)'
                        specific_match = re.search(very_simple, self.text, re.IGNORECASE)
                        logging.info(f"Tercer intento con patrón muy simple: {very_simple}")
                        
                    # Intento genérico para cualquier referencia a P1/P2/P3
                    if not specific_match:
                        generic_p = rf'[PE]{period_num}.*?(\d+[,.]\d+).*?[€]'
                        generic_match = re.search(generic_p, self.text, re.IGNORECASE)
                        if generic_match:
                            logging.info(f"Encontrada referencia genérica a {period_key} con valor: {generic_match.group(1)}")
                            try:
                                self.energy_prices[period_key] = float(generic_match.group(1).replace(',', '.'))
                                logging.info(f"Precio de energía para {period_key} detectado por referencia genérica: {self.energy_prices[period_key]} €/kWh")
                            except (ValueError, IndexError) as e:
                                logging.warning(f"Error al convertir valor de referencia genérica para {period_key}: {e}")
                    
                    if specific_match:
                        # Si hemos encontrado kWh x precio, extraemos el precio (segundo grupo)
                        try:
                            price = float(specific_match.group(2).replace(',', '.'))
                            self.energy_prices[period_key] = price
                            logging.info(f"Precio de energía para {period_key} detectado en formato específico: {price} €/kWh")
                            
                            # También podemos extraer el consumo del mismo patrón
                            consumption = float(specific_match.group(1).replace(',', '.').replace(' ', ''))
                            if not self.energy_consumption.get(period_key):
                                self.energy_consumption[period_key] = consumption
                                logging.info(f"Consumo para {period_key} extraído del patrón de precio: {consumption} kWh")
                        except (ValueError, IndexError) as e:
                            logging.warning(f"Error al extraer precio/consumo para {period_key} en formato específico: {str(e)}")
                            # Si falla el segundo grupo, intentamos con el primer grupo que podría ser el precio
                            try:
                                if len(specific_match.groups()) > 0:
                                    price = float(specific_match.group(1).replace(',', '.'))
                                    self.energy_prices[period_key] = price
                                    logging.info(f"Precio alternativo para {period_key}: {price} €/kWh (primer grupo)")
                            except (ValueError, IndexError) as e2:
                                logging.warning(f"También falló la extracción alternativa: {e2}")
"""

        # Ensamblar el archivo corregido
        fixed_content = before_section + fixed_section
        if end_found:
            fixed_content += after_section
        
        # Guardar el archivo corregido
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write(fixed_content)
        
        print(f"Archivo corregido guardado en: {file_path}")
        return True
        
    except Exception as e:
        print(f"Error al procesar el archivo: {str(e)}")
        # Restaurar desde la copia de seguridad
        shutil.copy2(backup_path, file_path)
        print("Se restauró el archivo original desde la copia de seguridad")
        return False

if __name__ == "__main__":
    tariff_detector_path = os.path.join('app', 'services', 'tariff_detector.py')
    if not os.path.exists(tariff_detector_path):
        print(f"No se encontró el archivo en: {tariff_detector_path}")
        sys.exit(1)
    
    success = fix_energy_price_section(tariff_detector_path)
    if success:
        print("Corrección completada con éxito.")
    else:
        print("No se pudo completar la corrección.")
