#!/usr/bin/env python
"""
Script para corregir la configuración de correo en el servidor.
Este script modifica la configuración de Flask-Mail para usar SSL o TLS correctamente,
pero no ambos al mismo tiempo.
"""

import os
import sys

# Configurar el entorno
os.environ['FLASK_ENV'] = 'production'

def fix_mail_config():
    """
    Corrige la configuración de correo para evitar el error STARTTLS.
    """
    from app import create_app
    
    # Crear la aplicación con la configuración de producción
    app = create_app('production')
    
    with app.app_context():
        # Verificar si estamos usando SSL o TLS
        use_ssl = app.config.get('MAIL_USE_SSL')
        use_tls = app.config.get('MAIL_USE_TLS')
        port = app.config.get('MAIL_PORT')
        
        print(f"\nConfiguración de correo actual:")
        print(f"MAIL_SERVER: {app.config.get('MAIL_SERVER')}")
        print(f"MAIL_PORT: {port}")
        print(f"MAIL_USE_SSL: {use_ssl}")
        print(f"MAIL_USE_TLS: {use_tls}")
        print(f"MAIL_TIMEOUT: {app.config.get('MAIL_TIMEOUT')}")
        
        # Si ambos están configurados, dar prioridad a SSL si el puerto es 465
        if use_ssl and use_tls:
            if port == 465:
                app.config['MAIL_USE_TLS'] = False
                print(f"\n⚠️ Configuración corregida: SSL habilitado, TLS deshabilitado (puerto {port})")
            else:
                app.config['MAIL_USE_SSL'] = False
                print(f"\n⚠️ Configuración corregida: TLS habilitado, SSL deshabilitado (puerto {port})")
        
        # Asegurarse de que el remitente predeterminado esté configurado
        if not app.config.get('MAIL_DEFAULT_SENDER'):
            app.config['MAIL_DEFAULT_SENDER'] = app.config.get('MAIL_USERNAME')
            print(f"\n⚠️ Remitente predeterminado configurado: {app.config['MAIL_DEFAULT_SENDER']}")
        
        # Aumentar el timeout si es necesario
        if app.config.get('MAIL_TIMEOUT', 0) < 30:
            app.config['MAIL_TIMEOUT'] = 30
            print(f"\n⚠️ Timeout aumentado a {app.config['MAIL_TIMEOUT']} segundos")
        
        # Mostrar la configuración actualizada
        print(f"\nConfiguración de correo actualizada:")
        print(f"MAIL_SERVER: {app.config.get('MAIL_SERVER')}")
        print(f"MAIL_PORT: {app.config.get('MAIL_PORT')}")
        print(f"MAIL_USE_SSL: {app.config.get('MAIL_USE_SSL')}")
        print(f"MAIL_USE_TLS: {app.config.get('MAIL_USE_TLS')}")
        print(f"MAIL_TIMEOUT: {app.config.get('MAIL_TIMEOUT')}")
        
        # Verificar la configuración del archivo .env
        print("\n🔍 Para solucionar permanentemente este problema, asegúrate de que tu archivo .env contenga:")
        if port == 465:
            print("MAIL_USE_SSL=True")
            print("# MAIL_USE_TLS=True  # Comenta o elimina esta línea")
        else:
            print("MAIL_USE_TLS=True")
            print("# MAIL_USE_SSL=True  # Comenta o elimina esta línea")
        
        print("\n📝 Instrucciones para corregir permanentemente el problema:")
        print("1. Edita el archivo .env en tu servidor")
        print("2. Asegúrate de que solo una de las opciones (SSL o TLS) esté habilitada")
        print("3. Reinicia la aplicación para aplicar los cambios")
        
        # Probar la conexión al servidor de correo
        try:
            from flask_mail import Mail
            mail = Mail(app)
            with mail.connect() as conn:
                print("\n✅ Conexión al servidor de correo exitosa!")
        except Exception as e:
            print(f"\n❌ Error al conectar con el servidor de correo: {str(e)}")
            print("\nVerifica la configuración de tu servidor de correo y asegúrate de que:")
            print("- El puerto 465 debe usar SSL (no TLS)")
            print("- El puerto 587 debe usar TLS (no SSL)")
            print("- El servidor de correo está accesible desde tu servidor")
            print("- Las credenciales son correctas")

if __name__ == '__main__':
    print("=== Herramienta de corrección de configuración de correo ===")
    fix_mail_config()
