#!/usr/bin/env python
"""
Script para diagnosticar problemas con la API de OpenAI.
Este script verifica la configuración de OpenAI y realiza una solicitud de prueba
para identificar problemas con la API.
"""

import os
import sys
import json
import requests
import logging
from dotenv import load_dotenv

# Configurar logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Cargar variables de entorno
load_dotenv()

def get_openai_api_key():
    """Obtener la clave de API de OpenAI de forma segura"""
    # Intentar obtener la clave de las variables de entorno
    env_key = os.getenv('OPENAI_API_KEY')
    if env_key:
        return env_key
    
    # Usar la clave hardcodeada como último recurso
    return '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'

def test_openai_api():
    """Realizar una solicitud de prueba a la API de OpenAI"""
    api_key = get_openai_api_key()
    
    if not api_key:
        logger.error("No se encontró la clave de API de OpenAI")
        return False
    
    logger.info(f"Usando clave de API: {api_key[:10]}...{api_key[-5:]}")
    
    url = "https://api.openai.com/v1/chat/completions"
    headers = {
        "Content-Type": "application/json",
        "Authorization": f"Bearer {api_key}"
    }
    
    payload = {
        "model": "gpt-3.5-turbo",
        "messages": [{"role": "user", "content": "Hola, esto es una prueba."}],
        "temperature": 0.7,
        "max_tokens": 50
    }
    
    try:
        logger.info("Enviando solicitud de prueba a OpenAI...")
        resp = requests.post(url, headers=headers, json=payload, timeout=30)
        resp.raise_for_status()
        data = resp.json()
        
        logger.info("Respuesta recibida correctamente")
        logger.info(f"Contenido: {data['choices'][0]['message']['content']}")
        return True
    except requests.exceptions.HTTPError as e:
        logger.error(f"Error HTTP: {e}")
        logger.error(f"Respuesta: {e.response.text}")
        return False
    except requests.exceptions.ConnectionError as e:
        logger.error(f"Error de conexión: {e}")
        return False
    except requests.exceptions.Timeout as e:
        logger.error(f"Timeout: {e}")
        return False
    except requests.exceptions.RequestException as e:
        logger.error(f"Error en la solicitud: {e}")
        return False
    except Exception as e:
        logger.error(f"Error inesperado: {e}")
        return False

def check_environment():
    """Verificar la configuración del entorno"""
    logger.info("Verificando configuración del entorno...")
    
    # Verificar variables de entorno
    logger.info(f"FLASK_ENV: {os.getenv('FLASK_ENV')}")
    logger.info(f"FLASK_CONFIG: {os.getenv('FLASK_CONFIG')}")
    logger.info(f"OPENAI_API_KEY configurada: {'Sí' if os.getenv('OPENAI_API_KEY') else 'No'}")
    
    # Verificar conectividad a Internet
    try:
        logger.info("Verificando conectividad a Internet...")
        requests.get("https://api.openai.com", timeout=5)
        logger.info("Conectividad a Internet: OK")
    except Exception as e:
        logger.error(f"Error de conectividad: {e}")
    
    # Verificar proxy
    logger.info(f"HTTP_PROXY: {os.getenv('HTTP_PROXY')}")
    logger.info(f"HTTPS_PROXY: {os.getenv('HTTPS_PROXY')}")

def fix_common_issues():
    """Sugerir soluciones para problemas comunes"""
    logger.info("\nSoluciones para problemas comunes:")
    logger.info("1. Asegúrate de que la clave de API de OpenAI esté configurada en el archivo .env")
    logger.info("2. Verifica que la aplicación tenga acceso a Internet")
    logger.info("3. Si estás usando gunicorn, asegúrate de que las variables de entorno estén disponibles para los workers")
    logger.info("4. Verifica que no haya un proxy bloqueando las conexiones a api.openai.com")
    logger.info("5. Asegúrate de que la clave de API no haya expirado o esté restringida")
    
    logger.info("\nPara ejecutar gunicorn con las variables de entorno correctas:")
    logger.info("screen -dmS ole_app bash -c \"cd /home/oleconsulting/public_html && source venv/bin/activate && export OPENAI_API_KEY='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' && export FLASK_CONFIG=production && gunicorn -w 4 -b 0.0.0.0:9000 run:app\"")

if __name__ == "__main__":
    print("=== Diagnóstico de OpenAI API ===")
    
    # Verificar entorno
    check_environment()
    
    # Probar API de OpenAI
    print("\n=== Probando conexión a OpenAI API ===")
    success = test_openai_api()
    
    if success:
        print("\n✅ La API de OpenAI funciona correctamente")
    else:
        print("\n❌ Se encontraron problemas con la API de OpenAI")
        fix_common_issues()
