#!/usr/bin/env python3
"""
Script de diagnóstico para la extracción OCR con DeepSeek API
"""
import os
import sys
import json
import logging
from dotenv import load_dotenv

# Configurar logging para ver detalles
logging.basicConfig(level=logging.DEBUG, format='%(asctime)s - %(levelname)s - %(message)s')

def main():
    print("=== DIAGNÓSTICO DE EXTRACCIÓN OCR ===\n")
    
    # 1. Verificar carga de variables de entorno
    print("1. Verificando carga de variables de entorno...")
    load_dotenv()
    
    # 2. Verificar API KEY
    print("2. Verificando DEEPSEEK_API_KEY...")
    api_key = os.getenv("DEEPSEEK_API_KEY")
    
    if not api_key:
        print("❌ ERROR: DEEPSEEK_API_KEY no encontrada en .env")
        print("   Solución: Añade 'DEEPSEEK_API_KEY=sk-tu_api_key' a tu archivo .env")
        return False
    elif api_key.startswith('sk-tu_deepseek'):
        print("❌ ERROR: DEEPSEEK_API_KEY parece ser el valor de ejemplo")
        print("   Solución: Reemplaza con tu API KEY real de DeepSeek")
        return False
    else:
        print(f"✅ API KEY encontrada: {api_key[:10]}...{api_key[-4:]}")
    
    # 3. Probar importación del módulo
    print("\n3. Probando importación del módulo llm_multi_provider...")
    try:
        from app.services.llm_multi_provider import extract_with_llm
        print("✅ Módulo importado correctamente")
    except Exception as e:
        print(f"❌ ERROR importando módulo: {e}")
        return False
    
    # 4. Probar conectividad básica con la API
    print("\n4. Probando conectividad con DeepSeek API...")
    try:
        response = call_deepseek_api(
            messages=[{"role": "user", "content": "Responde solo con: {'test': 'ok'}"}],
            model="deepseek-chat",
            temperature=0
        )
        
        if "error" in response:
            print(f"❌ ERROR en API: {response['error']}")
            
            if "inválida" in response.get('error', '').lower():
                print("   Posibles causas:")
                print("   - API KEY incorrecta o expirada")
                print("   - Formato de API KEY incorrecto")
                print("   - Cuenta de DeepSeek sin fondos/créditos")
            elif "límite" in response.get('error', '').lower():
                print("   Causa: Has excedido el límite de peticiones")
                print("   Solución: Espera un poco e inténtalo de nuevo")
                
            return False
        else:
            print("✅ API responde correctamente")
            print(f"   Respuesta: {json.dumps(response, indent=2)}")
    
    except Exception as e:
        print(f"❌ ERROR probando API: {e}")
        return False
    
    # 5. Probar extracción con texto de ejemplo
    print("\n5. Probando extracción con texto de ejemplo...")
    sample_text = """
    CUPS: ES0031607025821013ZW
    Cliente: JUAN PÉREZ GARCÍA
    Dirección: CALLE EJEMPLO 123, 28001 MADRID
    Fecha factura: 15/03/2024
    Periodo facturación: 15/02/2024 - 15/03/2024
    Potencia contratada: 4.4 kW
    Consumo P1: 150 kWh - Precio: 0.15 €/kWh
    """
    
    try:
        result = extract_with_llm(sample_text)
        if result:
            print("✅ Extracción exitosa")
            print(f"   Datos extraídos: {json.dumps(result, indent=2, ensure_ascii=False)}")
        else:
            print("❌ ERROR: La extracción devolvió None")
            print("   Revisa los logs anteriores para más detalles")
            return False
            
    except Exception as e:
        print(f"❌ ERROR en extracción: {e}")
        return False
    
    print("\n🎉 DIAGNÓSTICO COMPLETADO - Todo funciona correctamente!")
    return True

if __name__ == "__main__":
    success = main()
    if not success:
        print("\n📋 PASOS PARA SOLUCIONAR:")
        print("1. Verifica que tu archivo .env existe y contiene DEEPSEEK_API_KEY")
        print("2. Asegúrate de que la API KEY es válida y activa")
        print("3. Verifica que tu cuenta DeepSeek tiene créditos disponibles")
        print("4. Ejecuta este script de nuevo después de las correcciones")
        sys.exit(1)
    else:
        print("\n✅ El sistema OCR está configurado correctamente")
        sys.exit(0)
