import os
from flask_migrate import Migrate, init as migrate_init, migrate as migrate_create, upgrade as migrate_upgrade, downgrade as migrate_downgrade
from app import create_app, db
from app.models import User, Article, InvoiceLead

# Crear la aplicación usando la configuración del entorno o la predeterminada
app = create_app(os.getenv('FLASK_ENV') or 'default')
migrate = Migrate(app, db)

def init_migrations():
    """Inicializar el directorio de migraciones"""
    with app.app_context():
        migrate_init(directory='migrations')
    print("Directorio de migraciones inicializado en 'migrations/'")

def create_migration(message='Migration'):
    """Crear una nueva migración basada en los cambios detectados en los modelos"""
    with app.app_context():
        migrate_create(directory='migrations', message=message)
    print(f"Migración creada con mensaje: '{message}'")

def upgrade_db(revision='head'):
    """Actualizar la base de datos a la revisión especificada (por defecto la más reciente)"""
    with app.app_context():
        migrate_upgrade(directory='migrations', revision=revision)
    print(f"Base de datos actualizada a revisión: {revision}")

def downgrade_db(revision='-1'):
    """Revertir la base de datos a la revisión anterior"""
    with app.app_context():
        migrate_downgrade(directory='migrations', revision=revision)
    print(f"Base de datos revertida a revisión: {revision}")

def create_admin_user(username='admin', email='admin@oleconsulting.es', password='admin'):
    """Crear un usuario administrador en la base de datos"""
    with app.app_context():
        if not User.query.filter_by(username=username).first():
            user = User(username=username, email=email, is_admin=True)
            user.set_password(password)
            db.session.add(user)
            db.session.commit()
            print(f"Usuario administrador creado: {username}")
        else:
            print(f"El usuario {username} ya existe.")

if __name__ == '__main__':
    import sys
    if len(sys.argv) < 2:
        print("Uso: python db_migrate.py [init|migrate|upgrade|downgrade|create_admin] [mensaje|revisión]")
        sys.exit(1)
    
    command = sys.argv[1].lower()
    
    if command == 'init':
        init_migrations()
    elif command == 'migrate':
        message = sys.argv[2] if len(sys.argv) > 2 else 'Migration'
        create_migration(message)
    elif command == 'upgrade':
        revision = sys.argv[2] if len(sys.argv) > 2 else 'head'
        upgrade_db(revision)
    elif command == 'downgrade':
        revision = sys.argv[2] if len(sys.argv) > 2 else '-1'
        downgrade_db(revision)
    elif command == 'create_admin':
        username = sys.argv[2] if len(sys.argv) > 2 else 'admin'
        email = sys.argv[3] if len(sys.argv) > 3 else 'admin@oleconsulting.es'
        password = sys.argv[4] if len(sys.argv) > 4 else 'admin'
        create_admin_user(username, email, password)
    else:
        print(f"Comando desconocido: {command}")
        print("Uso: python db_migrate.py [init|migrate|upgrade|downgrade|create_admin] [mensaje|revisión]")
        sys.exit(1)
