import argparse
from datetime import datetime, timedelta

from app import create_app
from app.extensions import db
from app.models.user import User


def parse_args():
    parser = argparse.ArgumentParser(description="Limpiar usuarios sospechosos (sin rol e inactivos)")
    parser.add_argument("--days", type=int, default=30, help="Considerar usuarios creados en los últimos N días (default: 30)")
    parser.add_argument("--apply", action="store_true", help="Aplicar borrado (por defecto solo vista previa)")
    parser.add_argument("-y", "--yes", action="store_true", help="No preguntar confirmación al aplicar cambios")
    return parser.parse_args()


def main():
    args = parse_args()

    app = create_app()
    with app.app_context():
        cutoff = datetime.utcnow() - timedelta(days=args.days)

        # Usuarios sin roles y inactivos, creados recientemente
        q = (
            User.query
            .filter(~User.roles.any())
            .filter(User.is_active == False)  # noqa: E712 (comparación explícita con False)
            .filter(User.created_at >= cutoff)
        )

        candidates = q.all()
        print(f"Encontrados {len(candidates)} usuarios sin rol, inactivos y creados desde {cutoff:%Y-%m-%d}:")
        for u in candidates[:50]:  # mostrar hasta 50
            print(f" - id={u.id} username={u.username} email={u.email} created_at={u.created_at:%Y-%m-%d}")
        if len(candidates) > 50:
            print(f" ... y {len(candidates) - 50} más")

        if not args.apply:
            print("Vista previa completada. Ejecuta con --apply para eliminar.")
            return

        if not args.yes:
            confirm = input("¿Seguro que deseas eliminar estos usuarios? escribe 'ELIMINAR' para confirmar: ")
            if confirm.strip() != "ELIMINAR":
                print("Operación cancelada")
                return

        try:
            for u in candidates:
                db.session.delete(u)
            db.session.commit()
            print(f"Eliminados {len(candidates)} usuarios.")
        except Exception as e:
            db.session.rollback()
            print(f"Error al eliminar usuarios: {e}")


if __name__ == "__main__":
    main()
