#!/usr/bin/env python
"""
Script para revisar los logs del servidor y encontrar errores relacionados con OpenAI.
Este script busca errores específicos en los logs y proporciona sugerencias para resolverlos.
"""

import os
import sys
import re
import datetime

def find_recent_errors(log_file, hours=1, error_types=None):
    """
    Busca errores recientes en el archivo de log.
    
    Args:
        log_file: Ruta al archivo de log
        hours: Número de horas hacia atrás para buscar
        error_types: Lista de patrones de error a buscar
    """
    if error_types is None:
        error_types = [
            r"OpenAI",
            r"Error al generar",
            r"API key",
            r"Internal Server Error",
            r"Exception",
            r"Error"
        ]
    
    # Calcular la hora límite
    cutoff_time = datetime.datetime.now() - datetime.timedelta(hours=hours)
    
    # Patrones para fechas comunes en logs
    date_patterns = [
        r"\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2},\d{3})\]",  # [2025-07-02 12:21:33,123]
        r"(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2},\d{3})",      # 2025-07-02 12:21:33,123
        r"(\w{3} \w{3} \d{1,2} \d{2}:\d{2}:\d{2} \d{4})"      # Wed Jul 02 12:21:33 2025
    ]
    
    errors_found = []
    
    try:
        with open(log_file, 'r', encoding='utf-8', errors='ignore') as f:
            for line in f:
                # Intentar extraer la fecha del log
                log_date = None
                for pattern in date_patterns:
                    match = re.search(pattern, line)
                    if match:
                        try:
                            date_str = match.group(1)
                            if "," in date_str:  # Formato con milisegundos
                                log_date = datetime.datetime.strptime(date_str, "%Y-%m-%d %H:%M:%S,%f")
                            else:
                                # Intentar varios formatos comunes
                                for fmt in ["%a %b %d %H:%M:%S %Y", "%Y-%m-%d %H:%M:%S"]:
                                    try:
                                        log_date = datetime.datetime.strptime(date_str, fmt)
                                        break
                                    except ValueError:
                                        continue
                            break
                        except ValueError:
                            continue
                
                # Si no pudimos extraer la fecha, asumimos que es reciente
                is_recent = True if log_date is None else log_date >= cutoff_time
                
                # Si el log es reciente, buscar patrones de error
                if is_recent:
                    for pattern in error_types:
                        if re.search(pattern, line, re.IGNORECASE):
                            errors_found.append(line.strip())
                            # Incluir las siguientes líneas para contexto
                            for _ in range(5):
                                try:
                                    next_line = next(f, "").strip()
                                    if next_line:
                                        errors_found.append(next_line)
                                except StopIteration:
                                    break
                            errors_found.append("-" * 50)
                            break
        
        return errors_found
    except Exception as e:
        return [f"Error al leer el archivo de log: {str(e)}"]

def suggest_solutions(errors):
    """
    Analiza los errores y sugiere soluciones.
    """
    suggestions = []
    
    # Patrones comunes y sus soluciones
    error_patterns = {
        r"API key": "Verifica que la clave de API de OpenAI esté configurada correctamente en el archivo .env",
        r"STARTTLS": "Corrige la configuración de correo para usar SSL en lugar de TLS con el puerto 465",
        r"timeout": "Aumenta el timeout para las solicitudes a OpenAI o verifica la conexión a Internet",
        r"connection refused": "Verifica que no haya un firewall bloqueando las conexiones salientes",
        r"module.*not found": "Instala el módulo faltante con pip install",
        r"ImportError": "Verifica que todas las dependencias estén instaladas correctamente",
        r"NameError": "Hay una variable no definida en el código. Revisa las actualizaciones recientes.",
        r"AttributeError": "Hay un problema con un método o atributo. Verifica la versión de las bibliotecas.",
        r"KeyError": "Falta una clave en un diccionario. Verifica la respuesta de la API.",
        r"permission denied": "Verifica los permisos de los archivos y directorios"
    }
    
    for error in errors:
        for pattern, solution in error_patterns.items():
            if re.search(pattern, error, re.IGNORECASE):
                if solution not in suggestions:
                    suggestions.append(solution)
    
    # Añadir sugerencias generales si no encontramos nada específico
    if not suggestions:
        suggestions = [
            "Verifica que la clave de API de OpenAI esté configurada correctamente",
            "Asegúrate de que las variables de entorno estén disponibles para los workers de gunicorn",
            "Verifica los permisos de los archivos y directorios",
            "Revisa la conectividad a Internet desde el servidor",
            "Verifica que no haya un proxy bloqueando las conexiones a api.openai.com"
        ]
    
    return suggestions

def main():
    """Función principal"""
    print("=== Analizador de Logs para Errores de OpenAI ===")
    
    # Rutas comunes de logs
    log_paths = [
        "/home/oleconsulting/public_html/logs/app.log",
        "/home/oleconsulting/public_html/logs/error.log",
        "/var/log/apache2/error.log",
        "/var/log/nginx/error.log",
        "./logs/app.log",
        "./app.log",
        "./error.log"
    ]
    
    # Preguntar por la ruta del log
    print("\nRutas comunes de logs:")
    for i, path in enumerate(log_paths):
        print(f"{i+1}. {path}")
    
    print("\nIngrese el número de la ruta del log o escriba una ruta personalizada:")
    log_input = input("> ")
    
    try:
        idx = int(log_input) - 1
        if 0 <= idx < len(log_paths):
            log_file = log_paths[idx]
        else:
            log_file = log_input
    except ValueError:
        log_file = log_input
    
    # Verificar si el archivo existe
    if not os.path.exists(log_file):
        print(f"El archivo {log_file} no existe.")
        print("Por favor, ejecute este script en el servidor donde se encuentran los logs.")
        return
    
    # Buscar errores recientes
    print(f"\nBuscando errores recientes en {log_file}...")
    errors = find_recent_errors(log_file)
    
    if not errors:
        print("No se encontraron errores recientes relacionados con OpenAI.")
    else:
        print(f"\nSe encontraron {len(errors)} líneas de error:")
        for error in errors:
            print(error)
        
        # Sugerir soluciones
        print("\nPosibles soluciones:")
        solutions = suggest_solutions(errors)
        for i, solution in enumerate(solutions):
            print(f"{i+1}. {solution}")
    
    print("\nPara ejecutar una prueba de la API de OpenAI:")
    print("python debug_openai.py")

if __name__ == "__main__":
    main()
