/**
 * Sistema de carga condicional de scripts basado en el consentimiento de cookies
 * Este script se encarga de cargar/descargar scripts según las preferencias del usuario
 */

class ConditionalScripts {
    constructor() {
        this.cookieConsentName = 'ole_cookie_consent';
        this.loadedScripts = {
            analytics: [],
            marketing: []
        };
        this.scriptDefinitions = {
            analytics: [
                // Definición de scripts analíticos
                {
                    id: 'google-analytics',
                    src: 'https://www.googletagmanager.com/gtag/js?id=REEMPLAZAR-CON-ID-REAL',
                    async: true,
                    onload: function() {
                        window.dataLayer = window.dataLayer || [];
                        function gtag(){dataLayer.push(arguments);}
                        gtag('js', new Date());
                        gtag('config', 'REEMPLAZAR-CON-ID-REAL');
                    }
                }
                // Añadir más scripts analíticos aquí
            ],
            marketing: [
                // Definición de scripts de marketing
                {
                    id: 'facebook-pixel',
                    src: 'https://connect.facebook.net/en_US/fbevents.js',
                    async: true,
                    onload: function() {
                        // Código de inicialización del píxel de Facebook
                        // Se añadirá cuando se implemente realmente
                    }
                }
                // Añadir más scripts de marketing aquí
            ]
        };

        // Inicializar al cargar la página
        this.init();
    }

    /**
     * Inicializa el sistema y carga scripts según el consentimiento actual
     */
    init() {
        document.addEventListener('DOMContentLoaded', () => {
            this.updateScriptsBasedOnConsent();

            // Escuchar cambios en el consentimiento
            document.addEventListener('cookieConsentChanged', (event) => {
                console.log('Cambio en consentimiento detectado:', event.detail);
                this.updateScriptsBasedOnConsent();
            });
        });
    }

    /**
     * Obtiene el consentimiento actual de cookies
     * @returns {Object|null} Objeto de consentimiento o null si no hay consentimiento
     */
    getConsent() {
        const cookieValue = document.cookie
            .split('; ')
            .find(cookie => cookie.startsWith(this.cookieConsentName + '='));
        
        if (cookieValue) {
            try {
                const consentValue = decodeURIComponent(cookieValue.split('=')[1]);
                return JSON.parse(consentValue);
            } catch (e) {
                console.error('Error al parsear el consentimiento:', e);
                return null;
            }
        }
        
        return null;
    }

    /**
     * Actualiza la carga de scripts basada en el consentimiento actual
     */
    updateScriptsBasedOnConsent() {
        const consent = this.getConsent();
        
        if (!consent) {
            // Si no hay consentimiento, eliminamos todos los scripts no esenciales
            this.unloadAllScripts();
            return;
        }

        // Actualizar scripts analíticos
        if (consent.analytics) {
            this.loadScriptsByCategory('analytics');
        } else {
            this.unloadScriptsByCategory('analytics');
        }

        // Actualizar scripts de marketing
        if (consent.marketing) {
            this.loadScriptsByCategory('marketing');
        } else {
            this.unloadScriptsByCategory('marketing');
        }
    }

    /**
     * Carga scripts de una categoría específica
     * @param {string} category - Categoría de scripts ('analytics', 'marketing')
     */
    loadScriptsByCategory(category) {
        if (!this.scriptDefinitions[category]) return;
        
        this.scriptDefinitions[category].forEach(scriptDef => {
            // Verificar si el script ya está cargado
            if (this.loadedScripts[category].includes(scriptDef.id)) {
                return;
            }

            // Crear y añadir el script
            const script = document.createElement('script');
            script.id = scriptDef.id;
            script.src = scriptDef.src;
            script.async = scriptDef.async || false;
            
            if (scriptDef.onload) {
                script.onload = scriptDef.onload;
            }
            
            document.head.appendChild(script);
            
            // Registrar que el script ha sido cargado
            this.loadedScripts[category].push(scriptDef.id);
            console.log(`Script cargado: ${scriptDef.id} (${category})`);
        });
    }

    /**
     * Elimina scripts de una categoría específica
     * @param {string} category - Categoría de scripts ('analytics', 'marketing')
     */
    unloadScriptsByCategory(category) {
        if (!this.loadedScripts[category].length) return;
        
        this.loadedScripts[category].forEach(scriptId => {
            const script = document.getElementById(scriptId);
            if (script) {
                script.remove();
                console.log(`Script eliminado: ${scriptId} (${category})`);
            }
        });
        
        // Limpiar registro de scripts cargados
        this.loadedScripts[category] = [];
    }

    /**
     * Elimina todos los scripts no esenciales
     */
    unloadAllScripts() {
        this.unloadScriptsByCategory('analytics');
        this.unloadScriptsByCategory('marketing');
    }
}

// Inicializar el sistema de scripts condicionales
const conditionalScripts = new ConditionalScripts();
