import os
from flask import current_app, render_template
from flask_mail import Mail, Message

mail = Mail()

def init_app(app):
    """Inicializar la extensión de correo con la aplicación Flask"""
    # Importar y aplicar la corrección de configuración antes de inicializar mail
    from app.mail_config_fix import fix_mail_config
    fix_mail_config(app)
    
    # Inicializar mail con la configuración corregida
    mail.init_app(app)
    
    # Registrar información sobre la configuración de correo en los logs
    app.logger.info(f"Configuración de correo: SERVER={app.config.get('MAIL_SERVER')}, PORT={app.config.get('MAIL_PORT')}, SSL={app.config.get('MAIL_USE_SSL')}, TLS={app.config.get('MAIL_USE_TLS')}")
    
    # Verificar que no estén habilitados tanto SSL como TLS simultáneamente
    if app.config.get('MAIL_USE_SSL') and app.config.get('MAIL_USE_TLS'):
        app.logger.warning("Advertencia: Tanto SSL como TLS están habilitados en la configuración de correo, lo que puede causar errores.")
    

def send_email(subject, recipients, text_body, html_body=None, attachments=None):
    """
    Envía un correo electrónico
    
    Args:
        subject: Asunto del correo
        recipients: Lista de destinatarios
        text_body: Cuerpo del correo en formato texto
        html_body: Cuerpo del correo en formato HTML (opcional)
        attachments: Lista de archivos adjuntos (opcional)
                     Cada adjunto es un diccionario con las claves:
                     - filename: Nombre del archivo
                     - content_type: Tipo MIME del archivo
                     - data: Contenido binario del archivo
    """
    msg = Message(subject=subject, recipients=recipients)
    msg.body = text_body
    if html_body:
        msg.html = html_body
    
    # Adjuntar archivos si existen
    if attachments:
        for attachment in attachments:
            if isinstance(attachment, dict) and all(k in attachment for k in ['filename', 'content_type', 'data']):
                msg.attach(
                    filename=attachment['filename'],
                    content_type=attachment['content_type'],
                    data=attachment['data']
                )
    
    current_app.logger.info(f"Enviando correo a {recipients} con asunto: {subject}")
    mail.send(msg)

def send_lead_notification_email(lead, invoice_file_path=None):
    """
    Envía una notificación por correo cuando se recibe un nuevo lead de factura
    
    Args:
        lead: Objeto InvoiceLead con la información del lead
        invoice_file_path: Ruta al archivo de la factura subida (opcional)
    """
    # Obtener dirección de correo del admin desde la configuración
    admin_email = current_app.config.get('ADMIN_EMAIL') or 'info@grupo-ole.es'
    
    # Crear el asunto con información relevante
    subject = f"Nuevo estudio de factura - {lead.name} - {lead.cups or 'sin CUPS'}"
    
    # Crear el cuerpo del correo en formato texto
    text_body = f"""
    Se ha recibido un nuevo estudio de factura con los siguientes datos:
    
    ID: {lead.id}
    Nombre: {lead.name}
    Email: {lead.email}
    Teléfono: {lead.phone}
    CUPS: {lead.cups or 'No disponible'}
    Tarifa: {getattr(lead, 'tariff_type', 'No disponible')}
    Mensaje adicional: {lead.message or 'No hay mensaje adicional'}
    """
    
    # Crear el cuerpo del correo en formato HTML
    html_body = render_template('email/lead_notification.html', 
                               lead=lead,
                               details=lead.details)
    
    # Preparar adjuntos si existe el archivo de factura
    attachments = []
    if invoice_file_path and os.path.exists(invoice_file_path):
        with open(invoice_file_path, 'rb') as f:
            pdf_data = f.read()
            attachments.append({
                'filename': os.path.basename(invoice_file_path),
                'content_type': 'application/pdf',
                'data': pdf_data
            })
    
    # Enviar el correo
    send_email(
        subject=subject,
        recipients=[admin_email],
        text_body=text_body,
        html_body=html_body,
        attachments=attachments
    )
    
    current_app.logger.info(f"Correo de notificación enviado para lead ID: {lead.id}")

def send_contact_notification_email(form_data):
    """
    Envía una notificación por correo cuando se recibe un nuevo mensaje de contacto
    
    Args:
        form_data: Datos del formulario de contacto
    """
    # Destinatario del formulario de contacto
    contact_email = 'ahorro@oleconsulting.es'
    
    # Crear el asunto con información relevante
    subject = f"Nuevo mensaje de contacto - {form_data['name']}"
    
    # Crear el cuerpo del correo en formato texto
    text_body = f"""
    Se ha recibido un nuevo mensaje de contacto con los siguientes datos:
    
    Nombre: {form_data['name']}
    Email: {form_data['email']}
    Teléfono: {form_data['phone']}
    Asunto: {form_data['subject']}
    
    Mensaje:
    {form_data['message']}
    
    Acepta política de privacidad: Sí
    Suscripción newsletter: {'Sí' if form_data.get('newsletter') else 'No'}
    """
    
    # Crear el cuerpo del correo en formato HTML
    html_body = f"""
    <h2>Nuevo mensaje de contacto</h2>
    <p>Se ha recibido un nuevo mensaje de contacto con los siguientes datos:</p>
    <ul>
        <li><strong>Nombre:</strong> {form_data['name']}</li>
        <li><strong>Email:</strong> {form_data['email']}</li>
        <li><strong>Teléfono:</strong> {form_data['phone']}</li>
        <li><strong>Asunto:</strong> {form_data['subject']}</li>
    </ul>
    <h3>Mensaje:</h3>
    <p style="white-space: pre-wrap;">{form_data['message']}</p>
    <p><strong>Acepta política de privacidad:</strong> Sí</p>
    <p><strong>Suscripción newsletter:</strong> {'Sí' if form_data.get('newsletter') else 'No'}</p>
    """
    
    # Enviar el correo
    send_email(
        subject=subject,
        recipients=[contact_email],
        text_body=text_body,
        html_body=html_body
    )
    
    current_app.logger.info(f"Correo de contacto enviado de: {form_data['name']} ({form_data['email']})")

def send_savings_report_email(lead, savings_data, invoice_file_path=None):
    """
    Envía un reporte por correo con los resultados del cálculo de ahorro
    
    Args:
        lead: Objeto InvoiceLead con la información del lead
        savings_data: Diccionario con los datos de ahorro calculados
        invoice_file_path: Ruta al archivo de la factura subida (opcional)
    """
    # Obtener dirección de correo del admin desde la configuración
    admin_email = current_app.config.get('ADMIN_EMAIL') or 'info@grupo-ole.es'
    
    # Crear el asunto con información relevante
    subject = f"Resultados estudio factura - {lead.name} - {savings_data.get('yearly_savings', 0)} €/año"
    
    # Crear el cuerpo del correo en formato texto
    text_body = f"""
    Se han calculado los ahorros para el cliente:
    
    Cliente: {lead.name}
    Email: {lead.email}
    Teléfono: {lead.phone}
    CUPS: {lead.cups or 'No disponible'}
    
    Resultados del estudio:
    
    Ahorro anual: {savings_data.get('yearly_savings', 0)}€
    Porcentaje de ahorro: {savings_data.get('savings_percentage', 0)}%
    Coste actual: {savings_data.get('current_cost', 0)} €/año
    Coste estimado OLE: {savings_data.get('ole_cost', 0)} €/año
    Tarifa: {savings_data.get('tariff_type', '')}
    
    Datos adicionales en el archivo adjunto.
    """
    
    # Extraer datos detallados del lead
    try:
        details = lead.details or "{}"
        lead_details = eval(details)
    except:
        lead_details = {}
    
    # Crear el cuerpo del correo en formato HTML
    html_body = render_template('email/savings_report.html', 
                              lead=lead,
                              savings=savings_data,
                              details=lead_details)
    
    # Preparar adjuntos si existe el archivo de factura
    attachments = []
    if invoice_file_path and os.path.exists(invoice_file_path):
        with open(invoice_file_path, 'rb') as f:
            pdf_data = f.read()
            attachments.append({
                'filename': os.path.basename(invoice_file_path),
                'content_type': 'application/pdf',
                'data': pdf_data
            })
    
    # Enviar el correo
    send_email(
        subject=subject,
        recipients=[admin_email],
        text_body=text_body,
        html_body=html_body,
        attachments=attachments
    )
    
    current_app.logger.info(f"Correo de resultados enviado para lead ID: {lead.id}")
