"""
Módulo auxiliar para agregar al TariffDetector optimización en la extracción de consumos.
Contiene el método optimizado para extraer datos de consumo energético de facturas españolas.
"""

import re
import logging
import os

def extract_consumos_optimizado(self):
    """Método optimizado para extracción de consumos energéticos de facturas eléctricas españolas.
    Utiliza patrones más robustos y búsqueda más exhaustiva con mejor diagnóstico."""
    logging.info("Iniciando extracción OPTIMIZADA de consumos energéticos")
    
    # Verificación del texto
    if not self.text or len(self.text) < 100:
        logging.error(f"Texto insuficiente para extracción de consumos: {len(self.text)} caracteres")
        return
    
    # Guardar texto completo para análisis manual si es necesario
    try:
        with open('ultimo_texto_para_consumos.txt', 'w', encoding='utf-8') as f:
            f.write(f"TEXTO COMPLETO PARA ANÁLISIS DE CONSUMOS ({len(self.text)} caracteres):\n\n{self.text}")
        logging.info(f"Texto completo guardado para análisis manual de consumos")
    except Exception as e:
        logging.warning(f"No se pudo guardar texto para análisis manual: {e}")
    
    # Patrones optimizados por periodo - más flexibles y con mejor tolerancia a errores OCR
    patrones_optimizados = {
        # P1 - Punta
        'P1': [
            r'P1\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)', 
            r'[Pp]unta\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)',
            r'[Ee]nerg[íi]a\s*(?:activa)?\s*(?:en)?\s*[Pp]unta\s*[:=]?\s*([\d.,]+)',
            r'[Cc]onsumo\s*(?:en)?\s*(?:periodo)?\s*(?:P1|[Pp]unta)\s*[:=]?\s*([\d.,]+)',
            r'P1[^\d\n]{0,30}?([\d.,]+)\s*(?:kWh|KWH|kwh)', 
            r'[Pp]eriodo\s*1\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)'
        ],
        
        # P2 - Llano
        'P2': [
            r'P2\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)', 
            r'[Ll]lano\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)',
            r'[Ee]nerg[íi]a\s*(?:activa)?\s*(?:en)?\s*[Ll]lano\s*[:=]?\s*([\d.,]+)',
            r'[Cc]onsumo\s*(?:en)?\s*(?:periodo)?\s*(?:P2|[Ll]lano)\s*[:=]?\s*([\d.,]+)',
            r'P2[^\d\n]{0,30}?([\d.,]+)\s*(?:kWh|KWH|kwh)', 
            r'[Pp]eriodo\s*2\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)'
        ],
        
        # P3 - Valle
        'P3': [
            r'P3\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)', 
            r'[Vv]alle\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)',
            r'[Ee]nerg[íi]a\s*(?:activa)?\s*(?:en)?\s*[Vv]alle\s*[:=]?\s*([\d.,]+)',
            r'[Cc]onsumo\s*(?:en)?\s*(?:periodo)?\s*(?:P3|[Vv]alle)\s*[:=]?\s*([\d.,]+)',
            r'P3[^\d\n]{0,30}?([\d.,]+)\s*(?:kWh|KWH|kwh)', 
            r'[Pp]eriodo\s*3\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)'
        ]
    }
    
    # Patrones para consumo total (para casos donde no hay desglose por periodo)
    patrones_total = [
        r'[Tt]otal\s*(?:consumo|energía|energia)?\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)',
        r'[Ee]nerg[íi]a\s*(?:consumida|total)\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)',
        r'[Cc]onsumo\s*(?:total|energía|energia)?\s*[:=]?\s*([\d.,]+)\s*(?:kWh|KWH|kwh)',
        r'[Cc]onsumo\s*[^\d\n]{0,20}\s*([\d.,]+)\s*(?:kWh|KWH|kwh)',
        r'([\d.,]+)\s*(?:kWh|KWH|kwh)\s*(?:consumidos|facturados|totales)'
    ]
    
    # 1. Buscar consumos por periodo
    consumos_encontrados = False
    for periodo, patrones in patrones_optimizados.items():
        # Si ya tenemos este periodo, saltarlo
        if periodo in self.energy_consumption and self.energy_consumption[periodo] > 0:
            continue
            
        logging.info(f"Buscando consumo para periodo {periodo} con {len(patrones)} patrones optimizados")
        
        for i, patron in enumerate(patrones):
            try:
                # Buscar todas las coincidencias (no solo la primera)
                matches = re.finditer(patron, self.text, re.IGNORECASE | re.DOTALL)
                
                for match in matches:
                    try:
                        # Extraer valor y convertir
                        valor_str = match.group(1).replace(',', '.').strip()
                        valor = float(valor_str)
                        
                        # Validar valor razonable (0.1 a 10000 kWh)
                        if 0.1 <= valor <= 10000:
                            self.energy_consumption[periodo] = valor
                            logging.info(f"¡ÉXITO! Consumo {periodo}: {valor} kWh - Patrón {i+1}: '{match.group(0)}'")
                            consumos_encontrados = True
                            break
                    except (ValueError, IndexError) as e:
                        logging.debug(f"Error al convertir valor para {periodo}: {e} - Texto: '{match.group(0)}'")
                
                # Si ya encontramos este periodo, pasar al siguiente
                if periodo in self.energy_consumption and self.energy_consumption[periodo] > 0:
                    break
                    
            except Exception as e:
                logging.debug(f"Error aplicando patrón {i+1} para {periodo}: {e}")
    
    # 2. Si no se encontraron consumos por periodo, intentar con consumo total
    if not any(self.energy_consumption.values()):
        logging.warning("No se encontraron consumos por periodo. Buscando consumo total...")
        
        for i, patron in enumerate(patrones_total):
            try:
                matches = re.finditer(patron, self.text, re.IGNORECASE | re.DOTALL)
                
                for match in matches:
                    try:
                        valor_str = match.group(1).replace(',', '.').strip()
                        total_consumo = float(valor_str)
                        
                        # Validar valor razonable para consumo mensual
                        if 10 <= total_consumo <= 20000:
                            logging.info(f"Consumo total encontrado: {total_consumo} kWh - '{match.group(0)}'")
                            
                            # Distribuir según porcentajes típicos para 2.0TD
                            self.energy_consumption['P1'] = round(total_consumo * 0.33, 2)  # 33% en punta
                            self.energy_consumption['P2'] = round(total_consumo * 0.42, 2)  # 42% en llano
                            self.energy_consumption['P3'] = round(total_consumo * 0.25, 2)  # 25% en valle
                            
                            logging.info(f"Distribución estimada: P1={self.energy_consumption['P1']}, "
                                      f"P2={self.energy_consumption['P2']}, P3={self.energy_consumption['P3']} kWh")
                            logging.warning("NOTA: Consumos por periodo ESTIMADOS a partir del consumo total")
                            consumos_encontrados = True
                            break
                    except (ValueError, IndexError) as e:
                        logging.debug(f"Error al convertir consumo total: {e}")
                        
                if consumos_encontrados:
                    break
            except Exception as e:
                logging.debug(f"Error aplicando patrón de consumo total {i+1}: {e}")
    
    # 3. Último recurso - buscar cualquier número con kWh como diagnóstico
    if not consumos_encontrados:
        logging.error("NO SE ENCONTRARON CONSUMOS con ningún método. Realizando diagnóstico final...")
        
        try:
            # Buscar cualquier texto que contenga números y kWh
            texto_kwh = re.findall(r"[^\n]{0,50}([\d.,]+)\s*(?:kWh|KWH|kwh)[^\n]{0,50}", self.text, re.IGNORECASE)
            
            if texto_kwh:
                valores = []
                for i, fragmento in enumerate(texto_kwh[:10]):  # Mostrar hasta 10 fragmentos
                    logging.info(f"Posible consumo no detectado {i+1}: '{fragmento}'")
                    valores.append(fragmento)
                    
                # Guardar diagnóstico
                with open('diagnostico_consumos_fallidos.txt', 'w', encoding='utf-8') as f:
                    f.write("DIAGNÓSTICO DE CONSUMOS NO DETECTADOS\n\n")
                    f.write(f"Fragmentos con valores kWh encontrados:\n")
                    for i, v in enumerate(valores):
                        f.write(f"{i+1}. {v}\n")
            else:
                logging.error("No se encontró ningún valor numérico con kWh en el texto.")
                logging.error("Posible problema grave en la extracción OCR o formato de factura no soportado.")
        except Exception as e:
            logging.error(f"Error en diagnóstico final: {e}")
