from datetime import datetime
from flask import url_for
from werkzeug.security import generate_password_hash, check_password_hash
from flask_login import UserMixin
from app.extensions import db, login_manager
from .role import Role, user_roles # Importar Role y user_roles

class User(UserMixin, db.Model):
    __tablename__ = 'users'
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(64), index=True, unique=True, nullable=False)
    email = db.Column(db.String(120), index=True, unique=True, nullable=False)
    name = db.Column(db.String(100), nullable=True)  # Nombre completo del usuario
    profile_image = db.Column(db.String(255), nullable=True)  # Nombre del archivo de imagen de perfil
    password_hash = db.Column(db.String(256))
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    is_active = db.Column(db.Boolean, default=True)  # Nuevo campo para indicar si el usuario está activo
    # is_admin = db.Column(db.Boolean, default=False) # Campo eliminado

    # Relación con artículos (un autor puede tener muchos artículos)
    articles = db.relationship('Article', backref='author', lazy='dynamic')

    # Nueva relación muchos-a-muchos con Roles
    roles = db.relationship('Role', secondary=user_roles, back_populates='users', lazy='dynamic')

    def set_password(self, password):
        self.password_hash = generate_password_hash(password)

    def check_password(self, password):
        return check_password_hash(self.password_hash, password)

    # Helper para verificar si un usuario tiene un rol específico
    def has_role(self, role_name):
        return self.roles.filter_by(name=role_name).first() is not None

    @property
    def profile_image_url(self):
        if self.profile_image:
            try:
                return url_for('static', filename=f'uploads/profiles/{self.profile_image}')
            except RuntimeError:
                # Fuera de un contexto de aplicación, p.ej. en un shell
                return f'/static/uploads/profiles/{self.profile_image}'
        try:
            return url_for('static', filename='assets/img/avatars/default-avatar.svg', _external=False)
        except RuntimeError:
            return '/static/assets/img/avatars/default-avatar.svg'

    @property
    def first_name(self):
        if self.name:
            return self.name.split()[0]
        return self.username

    @property
    def last_name(self):
        if self.name and ' ' in self.name:
            return ' '.join(self.name.split()[1:])
        return ''

    def __repr__(self):
        return f'<User {self.username}>'

@login_manager.user_loader
def load_user(user_id):
    return User.query.get(int(user_id))
