from datetime import datetime
import json
from app.extensions import db

class InvoiceLead(db.Model):
    __tablename__ = 'invoice_leads'
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(100), nullable=False)
    email = db.Column(db.String(120), nullable=False)
    phone = db.Column(db.String(20), nullable=True)
    cups = db.Column(db.String(22), nullable=True) # CUPS suele tener 20 o 22 caracteres
    file_path = db.Column(db.String(255), nullable=False) # Ruta al archivo de factura subido
    message = db.Column(db.Text, nullable=True) # Mensaje adicional del usuario
    rgpd_consent = db.Column(db.Boolean, nullable=False, default=False)
    submitted_at = db.Column(db.DateTime, default=datetime.utcnow)
    status = db.Column(db.String(50), default='pending') # Ej: pending, processing, contacted, closed
    _details = db.Column(db.Text, nullable=True)  # Datos extraídos de la factura como JSON
    tariff_type = db.Column(db.String(20), nullable=True)  # Tipo de tarifa detectada (ej: 2.0TD, 3.0TD)
    created_by_user_id = db.Column(db.Integer, db.ForeignKey('users.id'), nullable=True)  # ID del comercial que creó el lead
    channel = db.Column(db.String(20), default='public')  # 'public' o 'commercial'
    
    # Campos para estudios de ahorro
    commercial_comments = db.Column(db.Text, nullable=True)  # Comentarios del comercial
    selected_plan = db.Column(db.String(255), nullable=True)  # Plan seleccionado (el de mayor ahorro)
    recommended_supplier = db.Column(db.String(255), nullable=True)  # Comercializadora recomendada
    savings_history = db.Column(db.Text, nullable=True)  # Histórico de estudios como JSON
    
    # Relación para obtener el usuario que creó el lead
    created_by = db.relationship('User', backref='created_leads', lazy=True)
    
    @property
    def details(self):
        """Deserializa los datos extraídos de la factura desde JSON"""
        if self._details:
            try:
                return json.loads(self._details)
            except json.JSONDecodeError:
                return {}
        return {}
    
    @details.setter
    def details(self, value):
        """Serializa los datos extraídos de la factura a JSON"""
        if value is None:
            self._details = None
            return
            
        # Verificar si el valor ya es una cadena JSON o un diccionario
        if isinstance(value, str):
            # Ya es una cadena JSON, guardarla directamente
            self._details = value
            # Intentar extraer la tarifa si está presente
            try:
                data_dict = json.loads(value)
                if isinstance(data_dict, dict) and 'tariff' in data_dict:
                    self.tariff_type = data_dict['tariff']
            except (json.JSONDecodeError, TypeError):
                # No es JSON válido o no contiene 'tariff'
                pass
        else:
            # Es un diccionario u otro objeto, convertirlo a JSON
            self._details = json.dumps(value)
            # Actualizar también el tipo de tarifa si está presente
            if isinstance(value, dict) and 'tariff' in value:
                self.tariff_type = value['tariff']
    
    @property
    def savings_history_data(self):
        """Deserializa el histórico de estudios de ahorro desde JSON"""
        if self.savings_history:
            try:
                return json.loads(self.savings_history)
            except json.JSONDecodeError:
                return []
        return []
    
    @savings_history_data.setter
    def savings_history_data(self, value):
        """Serializa el histórico de estudios de ahorro a JSON"""
        if value is None:
            self.savings_history = None
        else:
            self.savings_history = json.dumps(value)
    
    def add_savings_study(self, study_data):
        """Añade un nuevo estudio de ahorro al histórico"""
        history = self.savings_history_data
        study_data['timestamp'] = datetime.utcnow().isoformat()
        history.append(study_data)
        self.savings_history_data = history
    
    def __repr__(self):
        return f'<InvoiceLead {self.name} ({self.email})>'
