"""
Script para corregir la configuración de correo en la aplicación OLE-CONSULTING.
Este script modifica la configuración de Flask-Mail para usar SSL o TLS correctamente,
pero no ambos al mismo tiempo.
"""

from flask import Flask
import os
from dotenv import load_dotenv

def fix_mail_config(app):
    """
    Corrige la configuración de correo para evitar el error STARTTLS.
    
    Args:
        app: La aplicación Flask
    """
    # Verificar si estamos usando SSL o TLS
    use_ssl = app.config.get('MAIL_USE_SSL')
    use_tls = app.config.get('MAIL_USE_TLS')
    port = app.config.get('MAIL_PORT')
    
    # Si ambos están configurados, dar prioridad a SSL si el puerto es 465
    if use_ssl and use_tls:
        if port == 465:
            app.config['MAIL_USE_TLS'] = False
            print(f"Configuración corregida: SSL habilitado, TLS deshabilitado (puerto {port})")
        else:
            app.config['MAIL_USE_SSL'] = False
            print(f"Configuración corregida: TLS habilitado, SSL deshabilitado (puerto {port})")
    
    # Asegurarse de que el remitente predeterminado esté configurado
    if not app.config.get('MAIL_DEFAULT_SENDER'):
        app.config['MAIL_DEFAULT_SENDER'] = app.config.get('MAIL_USERNAME')
        print(f"Remitente predeterminado configurado: {app.config['MAIL_DEFAULT_SENDER']}")
    
    # Aumentar el timeout si es necesario
    if app.config.get('MAIL_TIMEOUT', 0) < 30:
        app.config['MAIL_TIMEOUT'] = 30
        print(f"Timeout aumentado a {app.config['MAIL_TIMEOUT']} segundos")
    
    return app

# Función para aplicar la corrección a la aplicación existente
def apply_fix():
    """Aplica la corrección a la aplicación Flask existente"""
    from app import create_app
    
    # Crear una instancia de la aplicación
    app = create_app()
    
    # Aplicar la corrección
    with app.app_context():
        fix_mail_config(app)
        
        # Mostrar la configuración actual
        print("\nConfiguración de correo actual:")
        print(f"MAIL_SERVER: {app.config.get('MAIL_SERVER')}")
        print(f"MAIL_PORT: {app.config.get('MAIL_PORT')}")
        print(f"MAIL_USE_TLS: {app.config.get('MAIL_USE_TLS')}")
        print(f"MAIL_USE_SSL: {app.config.get('MAIL_USE_SSL')}")
        print(f"MAIL_USERNAME: {app.config.get('MAIL_USERNAME')}")
        print(f"MAIL_DEFAULT_SENDER: {app.config.get('MAIL_DEFAULT_SENDER')}")
        print(f"MAIL_TIMEOUT: {app.config.get('MAIL_TIMEOUT')} segundos")

if __name__ == "__main__":
    apply_fix()
