from flask_wtf import FlaskForm
from flask_wtf.file import FileField, FileRequired, FileAllowed
from wtforms import StringField, TextAreaField, BooleanField, SubmitField
from wtforms.validators import DataRequired, Email, Optional, Length, ValidationError, Regexp

class InvoiceUploadForm(FlaskForm):
    """Formulario para subir una factura para análisis gratuito"""
    name = StringField('Nombre', validators=[DataRequired(), Length(min=2, max=50)])
    surname = StringField('Apellidos', validators=[DataRequired(), Length(min=2, max=50)])
    email = StringField('Correo electrónico', validators=[
        DataRequired(), 
        Email(message="Introduce un correo electrónico válido"),
        Length(max=100)
    ])
    phone = StringField('Número de contacto', validators=[
        DataRequired(),
        Length(min=9, max=9),
        Regexp('^[6-9][0-9]{8}$', message="Introduce un número válido sin +34 (móvil o fijo español)")
    ])
    invoice_file = FileField('Adjuntar factura (PDF - Máx. 10MB)', validators=[
        FileRequired(message='Debes subir un archivo de factura'),
        FileAllowed(['pdf'], message='Solo se permiten archivos PDF')
    ])
    invoice_photo = FileField('O sube una foto de tu factura (JPG/PNG - Máx. 10MB)', validators=[
        Optional(),
        FileAllowed(['jpg', 'jpeg', 'png'], message='Solo se permiten imágenes JPG o PNG')
    ])
    message = TextAreaField('Mensaje adicional', validators=[Optional(), Length(max=500)])
    rgpd_consent = BooleanField('Acepto la política de privacidad y el tratamiento de mis datos', 
                               validators=[DataRequired(message='Debes aceptar la política de privacidad')])
    submit = SubmitField('Solicitar Estudio Gratuito')
    
    def validate_invoice_file(self, field):
        # Validar tamaño máximo de 10MB para el archivo PDF
        max_size = 10 * 1024 * 1024  # 10MB en bytes
        if field.data and field.data.content_length > max_size:
            raise ValidationError(f"El archivo no puede superar los 10MB")
            
    def validate_invoice_photo(self, field):
        # Validar tamaño máximo de 10MB para la foto
        if field.data and field.data.content_length:
            max_size = 10 * 1024 * 1024  # 10MB en bytes
            if field.data.content_length > max_size:
                raise ValidationError(f"La imagen no puede superar los 10MB")
    
    # def validate_cups(self, cups):
    #     """Validación opcional del formato CUPS (si se proporciona)"""
    #     if cups.data and (len(cups.data) < 20 or len(cups.data) > 22):
    #         raise ValidationError('El código CUPS debe tener entre 20 y 22 caracteres')
