import os
from dotenv import load_dotenv

basedir = os.path.abspath(os.path.dirname(__file__))
# Carga .env desde la raíz del proyecto con override para asegurar precedencia
load_dotenv(os.path.join(basedir, '..', '.env'), override=True)

class Config:
    SECRET_KEY = os.environ.get('SECRET_KEY') or 'you-will-never-guess'
    # Extensiones de archivos permitidas para uploads
    ALLOWED_EXTENSIONS = {'pdf'}
    SQLALCHEMY_TRACK_MODIFICATIONS = False
    MAX_CONTENT_LENGTH = 10 * 1024 * 1024  # 10 MB para subida de archivos
    UPLOAD_FOLDER = os.path.join(basedir, 'static', 'uploads') # Asegúrate que esta carpeta exista o créala
    ALLOWED_EXTENSIONS = {'pdf', 'jpg', 'jpeg', 'png'}

    # Mail configuration
    MAIL_SERVER = os.environ.get('MAIL_SERVER')
    # Configuración SMTP
    MAIL_PORT = int(os.environ.get('MAIL_PORT') or 25)
    MAIL_USE_TLS = os.environ.get('MAIL_USE_TLS') is not None
    MAIL_USE_SSL = os.environ.get('MAIL_USE_SSL') is not None  # soporte para SSL directo (puerto 465)
    MAIL_TIMEOUT = int(os.environ.get('MAIL_TIMEOUT') or 10)  # segundos, evita bloqueos largos
    MAIL_USERNAME = os.environ.get('MAIL_USERNAME')
    MAIL_PASSWORD = os.environ.get('MAIL_PASSWORD')
    MAIL_DEFAULT_SENDER = os.environ.get('MAIL_DEFAULT_SENDER') or 'Ole Consulting <no-reply@oleconsulting.es>'
    ADMIN_EMAIL = os.environ.get('ADMIN_EMAIL') or 'info@oleconsulting.es'

    # reCAPTCHA v3
    RECAPTCHA_SITE_KEY = os.environ.get('RECAPTCHA_SITE_KEY')
    RECAPTCHA_SECRET_KEY = os.environ.get('RECAPTCHA_SECRET_KEY')
    RECAPTCHA_ENABLED = True # Puedes desactivarlo para testing

    # Feature flags / security hardening
    ENABLE_PUBLIC_REGISTRATION = os.environ.get('ENABLE_PUBLIC_REGISTRATION', 'false').lower() in ('1', 'true', 'yes')

    # Celery
    CELERY_BROKER_URL = os.environ.get('CELERY_BROKER_URL') or 'redis://localhost:6379/0'
    CELERY_RESULT_BACKEND = os.environ.get('CELERY_RESULT_BACKEND') or 'redis://localhost:6379/0'

    # Storage (local por defecto, S3 opcional)
    STORAGE_PROVIDER = os.environ.get('STORAGE_PROVIDER', 'local') # 'local' or 's3'
    S3_BUCKET_NAME = os.environ.get('S3_BUCKET_NAME')
    S3_REGION = os.environ.get('S3_REGION')
    AWS_ACCESS_KEY_ID = os.environ.get('AWS_ACCESS_KEY_ID')
    AWS_SECRET_ACCESS_KEY = os.environ.get('AWS_SECRET_ACCESS_KEY')

    @staticmethod
    def init_app(app):
        if not os.path.exists(app.config['UPLOAD_FOLDER']):
            os.makedirs(app.config['UPLOAD_FOLDER'])

class DevelopmentConfig(Config):
    DEBUG = True
    
    MYSQL_USER = os.environ.get('MYSQL_USER', 'root')
    MYSQL_PASSWORD = os.environ.get('MYSQL_PASSWORD', '')
    MYSQL_HOST = os.environ.get('MYSQL_HOST', 'localhost')
    MYSQL_PORT = os.environ.get('MYSQL_PORT', '3306')
    MYSQL_DB = os.environ.get('MYSQL_DB', 'ole_consulting_dev')
    
    SQLALCHEMY_DATABASE_URI = f"mysql://{MYSQL_USER}:{MYSQL_PASSWORD}@{MYSQL_HOST}:{MYSQL_PORT}/{MYSQL_DB}"
    
    SQLALCHEMY_ENGINE_OPTIONS = {
        'pool_recycle': 280,  # Reciclar conexiones cada ~4.5 minutos (evita el error 'MySQL server has gone away')
        'pool_pre_ping': True,  # Verificar conexiones antes de usarlas
        'pool_size': 10,  # Tamaño del pool de conexiones
        'max_overflow': 20  # Máximo número de conexiones adicionales
    }
    
    print(f"INFO: DevelopmentConfig using MySQL database at {MYSQL_HOST}:{MYSQL_PORT}/{MYSQL_DB}")

class TestingConfig(Config):
    TESTING = True
    SQLALCHEMY_DATABASE_URI = os.environ.get('TEST_DATABASE_URL') or \
        'sqlite:///:memory:' # Usa base de datos en memoria para tests
    WTF_CSRF_ENABLED = False # Deshabilita CSRF para tests de formularios
    RECAPTCHA_ENABLED = False
    MAIL_SUPPRESS_SEND = True # No enviar emails durante tests

class ProductionConfig(Config):
    MYSQL_USER = os.environ.get('MYSQL_USER', 'root')
    MYSQL_PASSWORD = os.environ.get('MYSQL_PASSWORD', '')
    MYSQL_HOST = os.environ.get('MYSQL_HOST', 'localhost')
    MYSQL_PORT = os.environ.get('MYSQL_PORT', '3306')
    MYSQL_DB = os.environ.get('MYSQL_DB', 'ole_consulting_dev')
    
    SQLALCHEMY_DATABASE_URI = f"mysql://{MYSQL_USER}:{MYSQL_PASSWORD}@{MYSQL_HOST}:{MYSQL_PORT}/{MYSQL_DB}"
    
    SQLALCHEMY_ENGINE_OPTIONS = {
        'pool_recycle': 280,  # Reciclar conexiones cada ~4.5 minutos (evita el error 'MySQL server has gone away')
        'pool_pre_ping': True,  # Verificar conexiones antes de usarlas
        'pool_size': 10,  # Tamaño del pool de conexiones
        'max_overflow': 20  # Máximo número de conexiones adicionales
    }
    
    print(f"INFO: ProductionConfig using MySQL database at {MYSQL_HOST}:{MYSQL_PORT}/{MYSQL_DB}")

    @classmethod
    def init_app(cls, app):
        Config.init_app(app)
        # Aquí podrías añadir configuraciones específicas de producción, como logging a un archivo o servicio.
        import logging
        from logging.handlers import SMTPHandler
        # Ejemplo de logging de errores por email (si MAIL_SERVER está configurado)
        if app.config.get('MAIL_SERVER'):
            auth = None
            if app.config['MAIL_USERNAME'] or app.config['MAIL_PASSWORD']:
                auth = (app.config['MAIL_USERNAME'], app.config['MAIL_PASSWORD'])
            secure = None
            if app.config['MAIL_USE_TLS']:
                secure = ()
            mail_handler = SMTPHandler(
                mailhost=(app.config['MAIL_SERVER'], app.config['MAIL_PORT']),
                fromaddr=app.config['MAIL_DEFAULT_SENDER'],
                toaddrs=[app.config['ADMIN_EMAIL']],
                subject='Error en Aplicación Olé Consulting',
                credentials=auth,
                secure=secure)
            mail_handler.setLevel(logging.ERROR)
            app.logger.addHandler(mail_handler)

config = {
    'development': DevelopmentConfig,
    'testing': TestingConfig,
    'production': ProductionConfig,
    'default': ProductionConfig
}
