from flask import render_template, request, Blueprint, abort
from app.models.article import Article
from sqlalchemy import desc
from . import blog_bp

# Conversión Markdown→HTML (fallback para artículos antiguos en BD)
try:
    import markdown  # La dependencia existe en requirements.txt
    _HAS_MD = True
except Exception:
    _HAS_MD = False

def _looks_like_html(text: str) -> bool:
    if not text:
        return False
    t = text.lower()
    # Heurística simple: presencia de etiquetas comunes o cierre de etiqueta
    return any(tag in t for tag in ['<p', '<h1', '<h2', '<h3', '<ul', '<ol', '<li', '<blockquote', '<strong', '<em', '</'])

def _markdown_to_html(content: str) -> str:
    if not content:
        return ''
    if not _HAS_MD:
        return content  # Si no hay markdown instalado, devolvemos tal cual para no romper
    try:
        return markdown.markdown(content, extensions=['extra', 'tables', 'nl2br'], output_format='html5')
    except Exception:
        # Fallback muy básico para no dejar el contenido roto
        return content.replace("<", "&lt;").replace(">", "&gt;").replace("\n", "<br>")

@blog_bp.route('/')
def list_articles():
    try:
        page = request.args.get('page', 1, type=int)
        order = request.args.get('order', 'newest')
        search = request.args.get('search', '')
        
        # Iniciar consulta con filtro de publicados
        query = Article.query.filter(Article.is_published == True)
        
        # Aplicar búsqueda si existe
        if search:
            search_term = f'%{search}%'
            query = query.filter(
                Article.title.ilike(search_term) | 
                Article.content.ilike(search_term) | 
                Article.summary.ilike(search_term)
            )
        
        # Aplicar ordenación
        if order == 'oldest':
            query = query.order_by(Article.published_at)  # Más antiguos primero
        else:  # 'newest' es el default
            query = query.order_by(desc(Article.published_at))  # Más nuevos primero
        
        # Configurar paginación para mostrar 9 artículos por página (1 destacado + 8 en grid)
        pagination = query.paginate(page=page, per_page=9, error_out=False)
        
        # Pasar los objetos Article directamente
        articles = pagination.items
        for article in articles:
            print(f"DEBUG: Article '{article.title}' featured_image: {article.featured_image}")
        
        return render_template('blog/list_articles.html', 
                              title='Blog', 
                              articles=articles,
                              pagination=pagination)
    except Exception as e:
        print(f"Error al cargar el listado del blog: {e}")
        # Usar placeholder solo en caso de error
        return render_template('blog/list_articles_placeholder.html', title='Blog')

@blog_bp.route('/<slug>')
def article_detail(slug):
    try:
        article = Article.query.filter_by(slug=slug, is_published=True).first_or_404()
        # Fallback: si el contenido parece Markdown (no HTML), convertir on-the-fly
        try:
            if not _looks_like_html(article.content):
                article.content = _markdown_to_html(article.content or '')
        except Exception:
            pass
        return render_template('blog/article_detail.html', 
                            title=article.title,
                            article=article)
    except Exception as e:
        print(f"Error al cargar artículo {slug}: {e}")
        abort(404) # Si hay algún error, mostrar 404
