from flask import render_template, redirect, url_for, request, flash
from app import db
from app.models.legal import LegalDocument
from . import admin_bp
from app.blueprints.admin.routes import admin_required
from flask_login import login_required
from flask_wtf import FlaskForm
from wtforms import StringField, TextAreaField, SelectField, BooleanField, SubmitField
from wtforms.validators import DataRequired, Length

class LegalDocumentForm(FlaskForm):
    """Formulario para crear y editar documentos legales."""
    document_type = SelectField('Tipo de Documento', choices=[
        ('cookies', 'Política de Cookies'),
        ('privacy', 'Política de Privacidad'),
        ('terms', 'Aviso Legal')
    ], validators=[DataRequired()])
    title = StringField('Título', validators=[DataRequired(), Length(min=3, max=200)])
    content = TextAreaField('Contenido', validators=[DataRequired()])
    version = StringField('Versión', default="1.0")
    is_active = BooleanField('Activo', default=True)
    submit = SubmitField('Guardar')

@admin_bp.route('/legal')
@login_required
@admin_required
def manage_legal_documents():
    """Lista todos los documentos legales."""
    try:
        documents = LegalDocument.query.all()
        return render_template('admin/legal/list.html', documents=documents)
    except Exception as e:
        flash(f'Error al cargar los documentos legales: {str(e)}', 'danger')
        return render_template('admin/legal/list.html', documents=[])

@admin_bp.route('/legal/new', methods=['GET', 'POST'])
@login_required
@admin_required
def create_legal_document():
    """Crea un nuevo documento legal."""
    form = LegalDocumentForm()
    
    if request.method == 'POST':
        # Obtener datos del formulario directamente
        document_type = request.form.get('document_type')
        title = request.form.get('title')
        content = request.form.get('content')
        version = request.form.get('version')
        is_active = 'is_active' in request.form
        
        # Validación manual de datos requeridos
        error = False
        if not document_type:
            flash('El tipo de documento es obligatorio.', 'danger')
            error = True
        if not title:
            flash('El título es obligatorio.', 'danger')
            error = True
        if not content:
            flash('El contenido es obligatorio.', 'danger')
            error = True
        
        if not error:
            try:
                # Verificar si ya existe un documento activo del mismo tipo
                existing_doc = LegalDocument.query.filter_by(
                    document_type=document_type,
                    is_active=True
                ).first()
                
                if existing_doc:
                    # Si hay uno existente, lo desactivamos para mantener solo uno activo por tipo
                    existing_doc.is_active = False
                    
                document = LegalDocument(
                    document_type=document_type,
                    title=title,
                    content=content,
                    version=version or "1.0",
                    is_active=is_active
                )
                
                # Depuración de datos
                print(f"Guardando documento: Tipo={document_type}, Título={title}, Contenido={content[:50]}..., Versión={version}, Activo={is_active}")
                
                db.session.add(document)
                db.session.commit()
                flash('Documento legal creado con éxito.', 'success')
                return redirect(url_for('admin.manage_legal_documents'))
            except Exception as e:
                db.session.rollback()
                print(f"Error al guardar documento legal: {str(e)}")
                flash(f'Error al crear el documento legal: {str(e)}', 'danger')
    
    return render_template('admin/legal/form.html', form=form, title='Nuevo documento legal')

@admin_bp.route('/legal/edit/<int:document_id>', methods=['GET', 'POST'])
@login_required
@admin_required
def edit_legal_document(document_id):
    """Edita un documento legal existente."""
    document = LegalDocument.query.get_or_404(document_id)
    form = LegalDocumentForm(obj=document)
    
    if form.validate_on_submit():
        try:
            # Si cambiamos el tipo y hay otro documento activo de ese tipo, desactivar el otro
            if form.is_active.data and form.document_type.data != document.document_type:
                existing_doc = LegalDocument.query.filter_by(
                    document_type=form.document_type.data,
                    is_active=True
                ).first()
                
                if existing_doc:
                    existing_doc.is_active = False
            
            form.populate_obj(document)
            db.session.commit()
            flash('Documento legal actualizado con éxito.', 'success')
            return redirect(url_for('admin.manage_legal_documents'))
        except Exception as e:
            db.session.rollback()
            flash(f'Error al actualizar el documento legal: {str(e)}', 'danger')
    
    return render_template('admin/legal/form.html', form=form, title='Editar documento legal')

@admin_bp.route('/legal/delete/<int:document_id>', methods=['POST'])
@login_required
@admin_required
def delete_legal_document(document_id):
    """Elimina un documento legal."""
    document = LegalDocument.query.get_or_404(document_id)
    try:
        db.session.delete(document)
        db.session.commit()
        flash('Documento legal eliminado con éxito.', 'success')
    except Exception as e:
        db.session.rollback()
        flash(f'Error al eliminar el documento legal: {str(e)}', 'danger')
    
    return redirect(url_for('admin.manage_legal_documents'))
