from flask import current_app
from app.models.role import Role
from app.models.user import User
from app.extensions import db
import click

def create_comercial_role():
    """Crea el rol 'Comercial' si no existe."""
    role = Role.query.filter_by(name='Comercial').first()
    if not role:
        role = Role(name='Comercial', description='Puede ver tarifas y leads, pero no puede modificar o eliminar registros.')
        db.session.add(role)
        db.session.commit()
        print("Rol 'Comercial' creado correctamente.")
    else:
        print("El rol 'Comercial' ya existe.")

def list_users():
    """Lista todos los usuarios y sus roles."""
    users = User.query.all()
    if not users:
        print("No hay usuarios registrados en la base de datos.")
        return
    
    print(f"\nTotal de usuarios: {len(users)}\n")
    print("{:<5} {:<30} {:<30} {:<10} {:<40}".format("ID", "Email", "Nombre", "Activo", "Roles"))
    print("-" * 100)
    
    for user in users:
        roles = ", ".join([role.name for role in user.roles]) if user.roles else "Sin rol"
        active_status = "Sí" if user.is_active else "No"
        print("{:<5} {:<30} {:<30} {:<10} {:<40}".format(
            user.id, 
            user.email[:28] + '..' if len(user.email) > 30 else user.email, 
            (user.username[:28] + '..') if user.username and len(user.username) > 30 else (user.username or 'Sin nombre'),
            active_status,
            roles
        ))

def assign_role_to_user(user_id, role_name):
    """Asigna un rol específico a un usuario."""
    user = User.query.get(user_id)
    if not user:
        print(f"Error: No se encontró ningún usuario con el ID {user_id}")
        return
    
    role = Role.query.filter_by(name=role_name).first()
    if not role:
        print(f"Error: El rol '{role_name}' no existe en la base de datos")
        return
    
    # Verificar si el usuario ya tiene ese rol
    if role in user.roles:
        print(f"El usuario {user.email} ya tiene el rol '{role_name}'")
        return
    
    # Asignar el rol al usuario
    user.roles.append(role)
    db.session.commit()
    print(f"Rol '{role_name}' asignado correctamente a {user.email}")
    

def migrate_english_roles():
    """Migra usuarios de roles en inglés a sus equivalentes en español y elimina los roles ingleses."""
    # Mapeo de roles inglés -> español
    role_mapping = {
        'Agent': 'Agente',
        'Price Analyst': 'Analista de Precios',
        'Super Admin': 'Super Administrador'
    }
    
    # Para cada mapeo de roles
    for english_role_name, spanish_role_name in role_mapping.items():
        english_role = Role.query.filter_by(name=english_role_name).first()
        spanish_role = Role.query.filter_by(name=spanish_role_name).first()
        
        if not english_role:
            print(f"El rol '{english_role_name}' no existe, se omite.")
            continue
        
        if not spanish_role:
            print(f"El rol '{spanish_role_name}' no existe. Se debe crear primero.")
            continue
        
        # Encontrar usuarios con el rol inglés
        affected_users = []
        for user in User.query.all():
            if user.has_role(english_role_name):
                affected_users.append(user)
        
        # Migrar usuarios al rol español
        print(f"\nMigrando usuarios del rol '{english_role_name}' a '{spanish_role_name}'...")
        for user in affected_users:
            # Asignar rol español
            if not user.has_role(spanish_role_name):
                user.roles.append(spanish_role)
                print(f"  - Asignado rol '{spanish_role_name}' a {user.username} ({user.email})")
            
            # Eliminar rol inglés
            user.roles.remove(english_role)
            print(f"  - Eliminado rol '{english_role_name}' de {user.username} ({user.email})")
        
        # Guardar cambios
        db.session.commit()
        
        # Eliminar el rol inglés
        try:
            db.session.delete(english_role)
            db.session.commit()
            print(f"Rol '{english_role_name}' eliminado de la base de datos.")
        except Exception as e:
            db.session.rollback()
            print(f"Error al eliminar el rol '{english_role_name}': {str(e)}")
    
    print("\nMigración completada.")
    
    # Listar todos los roles disponibles después de la migración
    print("\nRoles disponibles después de la migración:")
    for role in Role.query.all():
        print(f"  - {role.name}")


# Comandos CLI para Flask
def register_commands(app):
    """Registra los comandos para la aplicación Flask."""
    
    @app.cli.group()
    def roles():
        """Comandos para gestionar roles."""
        pass
    
    @roles.command('migrate-english')
    def migrate_english_roles_command():
        """Migra usuarios con roles en inglés a sus equivalentes en español."""
        click.echo('Iniciando migración de roles en inglés a español...')
        migrate_english_roles()
    
    @roles.command('list')
    def list_roles_command():
        """Lista todos los roles disponibles."""
        roles = Role.query.all()
        click.echo(f'Total de roles: {len(roles)}')
        for role in roles:
            users_count = role.users.count()
            click.echo(f'  - {role.name}: {users_count} usuarios')

